import { FILE_SIZE_LIMIT } from '@/configs';
import { BookTokenSubscriptionList } from '@/types/book-token';
import { normFile } from '@/utils';
import { UploadOutlined } from '@ant-design/icons';
import { Button, DatePicker, Form, Modal, Upload } from 'antd';
import { Dispatch, SetStateAction, useEffect } from 'react';
import { useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';
import { uploadProofMedia } from '@/services/book-token';
import dayjs from 'dayjs';
import { mediaResponseToFormValue } from '@/transformer/media/media';

interface ViewPaymentProps {
    open: boolean;
    setOpen: Dispatch<SetStateAction<boolean>>;
    selectedBookToken: BookTokenSubscriptionList | null;
}

const ViewPayment: React.FC<ViewPaymentProps> = ({ open, setOpen, selectedBookToken }) => {
    const { t } = useTranslation(['book-token', 'common', 'messages', 'layout']);
    const [uploadProofOfPaymentForm] = Form.useForm();

    const onCancelHandler = () => {
        uploadProofOfPaymentForm.resetFields();
        setOpen(false);
    };

    useEffect(() => {
        if (selectedBookToken) {
            uploadProofOfPaymentForm.setFieldsValue({
                paymentDate: dayjs.tz(selectedBookToken.paymentDate),
                uploadFile: selectedBookToken.bookTokenSubscriptionMedias.map((media) => mediaResponseToFormValue(media)),
            });
        }
    }, [selectedBookToken]);

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null}>
            <Form form={uploadProofOfPaymentForm} layout="vertical">
                <Form.Item name="paymentDate" label={t('payment-date')} rules={[{ required: true, message: t('messages:required') }]}>
                    <DatePicker className="w-full" size="large" disabled />
                </Form.Item>
                <Form.Item
                    name="uploadFile"
                    label={t('upload-file')}
                    rules={[{ required: true, message: t('messages:required') }]}
                    getValueFromEvent={normFile}
                    valuePropName="fileList"
                >
                    <Upload
                        multiple
                        className="!w-full"
                        maxCount={3}
                        beforeUpload={(file) => {
                            const isLTSizeLimit = file.size / 1024 / 1024 > FILE_SIZE_LIMIT;

                            if (isLTSizeLimit) {
                                toast.error(t('messages:file-too-large', { fileName: file.name, size: FILE_SIZE_LIMIT }));
                                return Upload.LIST_IGNORE;
                            }

                            return true;
                        }}
                        customRequest={(options) => uploadProofMedia('/member/book-token/upload', options)}
                        disabled
                    >
                        <Button block icon={<UploadOutlined />} size="large" disabled>
                            {t('common:upload-receipt')}
                        </Button>
                    </Upload>
                </Form.Item>
            </Form>
            <div className="flex flex-col md:flex-row gap-4">
                <Button block onClick={onCancelHandler}>
                    {t('common:close')}
                </Button>
            </div>
        </Modal>
    );
};

export default ViewPayment;
