import { Body, Controller, Get, Put } from '@nestjs/common';
import { Authenticator, Staff } from 'src/auth/auth.decorator';
import { StaffInRequest, User } from 'src/types';
import { UpdateProfileBody } from 'src/types/my-profile';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { StaffService } from '../staff/staff.service';
import { MyProfileService } from './my-profile.service';
import { updateProfileSchema } from './my-profile.validator';
import { ActivityLogService } from 'src/activity-log/activity-log.service';

@Controller()
export class MyProfileController {
    constructor(
        private readonly myProfileService: MyProfileService,
        private readonly staffService: StaffService,
        private activityLogService: ActivityLogService,
    ) {}

    @Authenticator(User.STAFF)
    @Get()
    async getMyProfile(@Staff() staff: StaffInRequest) {
        const staffData = await this.staffService.getStaffById(staff.id);

        return staffData;
    }

    @Authenticator(User.STAFF)
    @Put()
    async updateMyProfile(@Staff() staff: StaffInRequest, @Body(new JoiValidationPipe(updateProfileSchema)) body: UpdateProfileBody) {
        const { fullName } = staff;
        const staffData = await this.staffService.updateStaffProfile(staff.id, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'my-profile',
                id: staff.id,
            },
            staffId: staff.id,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-my-profile',
            data: staffData,
        });

        return staffData;
    }

    @Authenticator(User.STAFF)
    @Put('password')
    async updateMyPassword(@Staff() staff: StaffInRequest, @Body('password') password: string) {
        const { fullName } = staff;
        const staffData = await this.myProfileService.updateMyPassword(staff.id, password);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'my-profile',
                id: staff.id,
            },
            staffId: staff.id,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-my-profile-password',
            data: staffData,
        });

        return staffData;
    }

    @Authenticator(User.STAFF)
    @Put('email')
    async updateMyEmail(@Staff() staff: StaffInRequest, @Body('email') email: string) {
        const { fullName } = staff;
        const staffData = await this.staffService.updateStaffEmail(staff.id, { email });

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'my-profile',
                id: staff.id,
            },
            staffId: staff.id,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-my-profile-email',
            data: staffData,
        });

        return staffData;
    }

    @Authenticator(User.STAFF)
    @Put('phone')
    async updateMyPhone(@Staff() staff: StaffInRequest, @Body('phone') phone: string) {
        const { fullName } = staff;
        const staffData = await this.staffService.updateStaffPhoneNumber(staff.id, { phoneNumber: phone });

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'my-profile',
                id: staff.id,
            },
            staffId: staff.id,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-my-profile-phone-number',
            data: staffData,
        });

        return staffData;
    }
}
