import { Body, Controller, Delete, Get, HttpCode, HttpStatus, Param, Post, Put, Query } from '@nestjs/common';
import { MemberSharedService } from 'src/member/member-shared.service';
import { memberEmailSchema, createMemberSchema } from 'src/member/member-shared.validator';
import { Authenticator, Authorizer, Staff } from 'src/auth/auth.decorator';
import { StaffInRequest, User } from 'src/types';
import { MemberQuery } from 'src/types/member';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { MemberService } from './member.service';
import {
    memberQuerySchema,
    updateMemberEmailSchema,
    updateMemberPasswordSchema,
    updateMemberPhoneNumberSchema,
    updateMemberProfileSchema,
} from './member.validator';
import { ActivityLogService } from 'src/activity-log/activity-log.service';

@Controller()
export class MemberController {
    constructor(
        private memberService: MemberService,
        private memberSharedService: MemberSharedService,
        private activityLogService: ActivityLogService,
    ) {}

    @Authenticator(User.STAFF)
    @Authorizer('MEMBER_VIEW')
    @Get('')
    @HttpCode(HttpStatus.OK)
    async getMembers(@Query(new JoiValidationPipe(memberQuerySchema)) query: MemberQuery) {
        const memberList = await this.memberService.getMemberList(query);
        return memberList;
    }

    @Authenticator(User.STAFF)
    @Get('export')
    @HttpCode(HttpStatus.OK)
    async exportMembers() {
        const memberList = await this.memberService.getExportMemberList();
        return memberList;
    }

    @Authenticator(User.STAFF)
    @Authorizer('MEMBER_VIEW')
    @Get(':memberId')
    @HttpCode(HttpStatus.OK)
    async getMemberById(@Param('memberId') memberId: string) {
        const member = await this.memberService.getMemberById(memberId);
        return member;
    }

    @Authenticator(User.STAFF)
    @Authorizer('MEMBER_CREATE')
    @Post('')
    @HttpCode(HttpStatus.OK)
    async createMember(@Staff() staff: StaffInRequest, @Body(new JoiValidationPipe(createMemberSchema)) body: CreateMemberDto) {
        const { id: staffId, fullName } = staff;
        const createdMember = await this.memberSharedService.createMember(body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'member',
                id: createdMember.id,
            },
            staffId,
            executorName: fullName,
            action: 'CREATE',
            description: 'api-messages:create-member',
            data: createdMember,
        });

        return createdMember;
    }

    @Authenticator(User.STAFF)
    @Authorizer('MEMBER_UPDATE')
    @Put(':memberId/status')
    @HttpCode(HttpStatus.OK)
    async updateMemberStatus(@Staff() staff: StaffInRequest, @Param('memberId') memberId: string) {
        const { id: staffId, fullName } = staff;
        const response = await this.memberService.updateMemberStatus(memberId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'member',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-member-status',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('MEMBER_UPDATE')
    @Put(':memberId/profile')
    @HttpCode(HttpStatus.OK)
    async updateMemberProfile(
        @Staff() staff: StaffInRequest,
        @Param('memberId') memberId: string,
        @Body(new JoiValidationPipe(updateMemberProfileSchema)) body: UpdateMemberProfileDto,
    ) {
        const { id: staffId, fullName } = staff;
        const response = await this.memberService.updateMemberProfile(memberId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'member',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-member-profile',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('MEMBER_UPDATE')
    @Put(':memberId/email')
    @HttpCode(HttpStatus.OK)
    async updateMemberEmail(
        @Staff() staff: StaffInRequest,
        @Param('memberId') memberId: string,
        @Body(new JoiValidationPipe(updateMemberEmailSchema)) body: UpdateMemberEmailDto,
    ) {
        const { id: staffId, fullName } = staff;
        const response = await this.memberService.updateMemberEmail(memberId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'member',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-member-email',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('MEMBER_UPDATE')
    @Put(':memberId/phone-number')
    @HttpCode(HttpStatus.OK)
    async updateMemberPhoneNumber(
        @Staff() staff: StaffInRequest,
        @Param('memberId') memberId: string,
        @Body(new JoiValidationPipe(updateMemberPhoneNumberSchema)) body: UpdateMemberPhoneNumberDto,
    ) {
        const { id: staffId, fullName } = staff;
        const response = await this.memberService.updateMemberPhoneNumber(memberId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'member',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-member-phone-number',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('MEMBER_UPDATE')
    @Put(':memberId/password')
    @HttpCode(HttpStatus.OK)
    async updateMemberPassword(
        @Staff() staff: StaffInRequest,
        @Param('memberId') memberId: string,
        @Body(new JoiValidationPipe(updateMemberPasswordSchema)) body: UpdateMemberPasswordDto,
    ) {
        const { id: staffId, fullName } = staff;
        const response = await this.memberService.updateMemberPassword(memberId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'member',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-member-password',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('MEMBER_UPDATE')
    @Post('resend-verification-email')
    @HttpCode(HttpStatus.OK)
    async resendVerificationEmail(@Body(new JoiValidationPipe(memberEmailSchema)) body: MemberEmailDto) {
        return await this.memberSharedService.resendVerificationEmail(body.email);
    }

    @Authenticator(User.STAFF)
    @Authorizer('MEMBER_DELETE')
    @Delete(':memberId')
    @HttpCode(HttpStatus.OK)
    async deleteMember(@Staff() staff: StaffInRequest, @Param('memberId') memberId: string) {
        const { id: staffId, fullName } = staff;
        const response = await this.memberService.deleteMember(memberId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'member',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:delete-member',
            data: response,
        });

        return response;
    }
}
