import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { MedicalCertificateStatus, Prisma } from '@prisma/client';
import { PrismaService } from 'src/prisma/prisma.service';
import { Pagination } from 'src/types';

@Injectable()
export class MedicalCertificateService {
    constructor(private prisma: PrismaService) {}

    async getAllMedicalCertificate(query: Pagination) {
        const { page, pageSize, sortField, sortOrder } = query;

        const medicalCertificateWhereOptions: Prisma.MedicalCertificateWhereInput = {
            deletedAt: null,
        };

        const medicalCertificateCount = await this.prisma.medicalCertificate.count({
            where: medicalCertificateWhereOptions,
        });

        const currentPage = this.prisma.pageCounter(medicalCertificateCount, page, pageSize);

        const medicalCertificate = await this.prisma.medicalCertificate.findMany({
            where: medicalCertificateWhereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: {
                id: true,
                memberId: true,

                member: {
                    select: {
                        fullName: true,
                    },
                },
                studyGroupTaskId: true,
                studyGroupTask: {
                    select: {
                        studyGroupId: true,
                        studyGroup: {
                            select: {
                                name: true,
                            },
                        },
                    },
                },
                applyDate: true,
                status: true,
                media: {
                    select: {
                        mediaId: true,
                        media: {
                            select: {
                                key: true,
                                name: true,
                            },
                        },
                    },
                },
            },
        });

        return {
            total: medicalCertificateCount,
            rows: medicalCertificate,
            page: currentPage,
        };
    }

    async approveMedicalCertificate(id: string) {
        const medicalCertificate = await this.prisma.medicalCertificate.findUnique({
            where: {
                id,
            },
            select: {
                id: true,
                status: true,
            },
        });

        if (!medicalCertificate) {
            throw new HttpException('api-messages:medical-certificate-not-found', HttpStatus.NOT_FOUND);
        }

        if (medicalCertificate.status !== MedicalCertificateStatus.PENDING) {
            throw new HttpException('api-messages:medical-certificate-not-pending', HttpStatus.BAD_REQUEST);
        }

        const response = await this.prisma.medicalCertificate.update({
            where: {
                id: medicalCertificate.id,
            },
            data: {
                status: MedicalCertificateStatus.APPROVED,
            },
        });

        return response;
    }

    async rejectMedicalCertificate(id: string) {
        const medicalCertificate = await this.prisma.medicalCertificate.findUnique({
            where: {
                id,
            },
            select: {
                id: true,
                status: true,
            },
        });

        if (!medicalCertificate) {
            throw new HttpException('api-messages:medical-certificate-not-found', HttpStatus.NOT_FOUND);
        }

        if (medicalCertificate.status !== MedicalCertificateStatus.PENDING) {
            throw new HttpException('api-messages:medical-certificate-not-pending', HttpStatus.BAD_REQUEST);
        }

        const response = await this.prisma.medicalCertificate.update({
            where: {
                id: medicalCertificate.id,
            },
            data: {
                status: MedicalCertificateStatus.REJECTED,
            },
        });

        return response;
    }
}
