import { Controller, Get, Param, Put, Query } from '@nestjs/common';
import { MedicalCertificateService } from './medical-certificate.service';
import { Authenticator, Staff } from 'src/auth/auth.decorator';
import { Pagination, StaffInRequest, User } from 'src/types';
import { paginationSchema } from 'src/utils/validators/pagination.schema';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { ActivityLogService } from 'src/activity-log/activity-log.service';

@Controller()
export class MedicalCertificateController {
    constructor(
        private medicalCertificateService: MedicalCertificateService,
        private activityLogService: ActivityLogService,
    ) {}

    @Authenticator(User.STAFF)
    @Get('')
    async getAllMedicalCertificate(@Query(new JoiValidationPipe(paginationSchema)) query: Pagination) {
        const medicalCertificate = await this.medicalCertificateService.getAllMedicalCertificate(query);
        return medicalCertificate;
    }

    @Authenticator(User.STAFF)
    @Put(':medicalCertificateId/approve')
    async approveMedicalCertificate(@Staff() staff: StaffInRequest, @Param('medicalCertificateId') medicalCertificateId: string) {
        const { id: staffId, fullName } = staff;
        const response = await this.medicalCertificateService.approveMedicalCertificate(medicalCertificateId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'medical-certificate',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:approve-medical-certificate',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Put(':medicalCertificateId/reject')
    async rejectMedicalCertificate(@Staff() staff: StaffInRequest, @Param('medicalCertificateId') medicalCertificateId: string) {
        const { id: staffId, fullName } = staff;
        const response = await this.medicalCertificateService.rejectMedicalCertificate(medicalCertificateId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'medical-certificate',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:reject-medical-certificate',
            data: response,
        });

        return response;
    }
}
