import { Controller, Get, HttpCode, HttpStatus, Query } from '@nestjs/common';
import { Member, Role } from '@prisma/client';
import { Authenticator } from 'src/auth/auth.decorator';
import { CacheService } from 'src/cache/cache.service';
import { DataService } from 'src/data/data.service';
import { MemberDataQuery, User } from 'src/types';
import { MemberService } from '../member/member.service';
import { StudyRequestService } from '../study-request/study-request.service';

@Controller()
export class StaffDataController {
    constructor(
        private dataService: DataService,
        private cacheService: CacheService,
        private memberService: MemberService,
        private studyRequestService: StudyRequestService,
    ) {}

    @Authenticator(User.STAFF)
    @Get('role')
    @HttpCode(HttpStatus.OK)
    async getRoleData() {
        let roles: Partial<Role>[];

        const cachedRole: Partial<Role>[] | undefined = await this.cacheService.get('role');
        if (!cachedRole) {
            roles = await this.dataService.getRoleData();

            await this.cacheService.set('role', roles);
        } else {
            roles = cachedRole;
        }

        return roles;
    }

    @Authenticator(User.STAFF)
    @Get('birthday')
    @HttpCode(HttpStatus.OK)
    async getMemberBirthday() {
        const memberBirthday = await this.memberService.getMemberBirthdayList();
        return memberBirthday;
    }

    @Authenticator(User.STAFF)
    @Get('member')
    @HttpCode(HttpStatus.OK)
    async getMemberData(@Query() query: MemberDataQuery) {
        let member: Partial<Member>[];
        let cachedKey = 'member';

        if (query.keyword) {
            cachedKey += query.keyword;
        }

        const cachedMember: Partial<Member>[] | undefined = await this.cacheService.get(cachedKey);
        if (!cachedMember) {
            member = await this.dataService.getMemberData();

            await this.cacheService.set(cachedKey, member);
        } else {
            member = cachedMember;
        }

        return member;
    }

    @Authenticator(User.STAFF)
    @Get('request-list')
    @HttpCode(HttpStatus.OK)
    async getStudyRequestList() {
        const requestList = await this.studyRequestService.getAllStudyRequest();
        return requestList;
    }

    @Authenticator(User.STAFF)
    @Get('medical-certificate-list')
    @HttpCode(HttpStatus.OK)
    async getMedicalCertificateData() {
        const medicalCertificate = await this.dataService.getMedicalCertificateData();
        return medicalCertificate;
    }

    @Authenticator(User.STAFF)
    @Get('book-token-request-count')
    @HttpCode(HttpStatus.OK)
    async getBookTokenRequestCount() {
        const bookTokenRequestCount = await this.dataService.getBookTokenRequestCount();
        return bookTokenRequestCount;
    }
}
