import { Body, Controller, Get, HttpCode, HttpStatus, Param, Post, Res } from '@nestjs/common';
import { Throttle, seconds } from '@nestjs/throttler';
import { Response } from 'express';
import { COOKIE_DOMAIN, COOKIE_OPTIONS, USER_TOKEN_COOKIE_NAME } from 'src/config';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { AuthService } from './auth.service';
import { authenticationSchema, loginSchema } from './auth.validator';
import { staffEmailSchema } from '../staff/staff.validator';
import { StaffAuthenticateDto, StaffLoginDto } from './auth.dto';
import { StaffEmailDto } from '../staff/staff.dto';
import { StaffSharedService } from '../staff-shared.service';

@Controller()
export class AuthController {
    constructor(
        private authService: AuthService,
        private sharedService: StaffSharedService,
    ) {}

    @Post('authenticate')
    async authenticate(@Body(new JoiValidationPipe(authenticationSchema)) body: StaffAuthenticateDto) {
        const staff = await this.authService.authenticate(body.token, body.permission);
        return staff;
    }

    @Throttle({ default: { limit: 5, ttl: seconds(15) } })
    @Post('login')
    async login(@Body(new JoiValidationPipe(loginSchema)) body: StaffLoginDto, @Res() res: Response) {
        const token = await this.authService.login(body);
        res.cookie(USER_TOKEN_COOKIE_NAME.staff, token, COOKIE_OPTIONS).status(200).json({
            success: true,
            message: 'api-messages:login-success',
        });
    }

    @Post('logout')
    async logout(@Res() res: Response) {
        res.clearCookie(USER_TOKEN_COOKIE_NAME.staff, { domain: COOKIE_DOMAIN });
        res.status(200).json({
            success: true,
            message: 'api-messages:logout-success',
        });
    }

    @Get('tokenVerifier/:staffId/:token')
    async tokenVerifier(@Param('staffId') staffId: string, @Param('token') token: string) {
        const staff = await this.authService.tokenVerifier(staffId, token);
        return staff;
    }

    @Post('verify/:staffId/:token')
    async verifyStaff(@Param('staffId') staffId: string, @Param('token') token: string, @Body() body: { password: string }) {
        const { password } = body;
        return await this.authService.verifyStaff({ id: staffId, password, token });
    }

    @Post('forgot-password')
    async forgotPassword(@Body(new JoiValidationPipe(staffEmailSchema)) body: StaffEmailDto) {
        const { email } = body;
        return await this.authService.forgotPassword(email);
    }

    @Get('verify-reset-token/:staffId/:token')
    async verifyResetToken(@Param('staffId') staffId: string, @Param('token') token: string) {
        const response = await this.authService.verifyResetToken(staffId, token);
        return response;
    }

    @Post('reset-password/:staffId/:token')
    async resetPassword(@Param('staffId') staffId: string, @Param('token') token: string, @Body('password') password: string) {
        const response = await this.authService.resetPassword(staffId, token, password);
        return response;
    }

    @Post('resend-verification-email')
    @HttpCode(HttpStatus.OK)
    async resendVerificationEmail(@Body(new JoiValidationPipe(staffEmailSchema)) body: StaffEmailDto) {
        await this.sharedService.resendVerificationEmail(body.email);
        return {
            success: true,
            message: 'api-messages:verification-email-sent',
        };
    }
}
