import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { PackageStatus } from '@prisma/client';
import { PrismaService } from 'src/prisma/prisma.service';

@Injectable()
export class BillService {
    constructor(private prisma: PrismaService) {}

    async getStaffQuotationById() {
        const allPackage = await this.prisma.package.findMany({
            where: {
                deletedAt: null,
                status: PackageStatus.ACTIVE,
            },
        });

        return { allPackage };
    }

    async getMemberQuotationById(memberId: string) {
        const allPackage = await this.prisma.package.findMany({
            where: {
                deletedAt: null,
                status: PackageStatus.ACTIVE,
            },
        });

        const member = await this.prisma.member.findUnique({
            where: {
                id: memberId,
            },
            select: {
                fullName: true,
                phoneNumber: true,
                address: true,
            },
        });

        return { allPackage, member };
    }

    async getInvoiceBySubscriptionId(bookTokenSubscriptionId: string) {
        const invoice = await this.prisma.bookTokenSubscription.findFirst({
            where: {
                id: bookTokenSubscriptionId,
            },
            select: {
                ...this.prisma.createSelect(['quantity', 'price', 'displayId', 'createdAt', 'paymentStatus', 'updatedAt']),
                member: {
                    select: this.prisma.createSelect(['fullName', 'preferredName', 'phoneNumber', 'address']),
                },
                package: {
                    select: this.prisma.createSelect(['name', 'description', 'price']),
                },
            },
        });

        if (!invoice) {
            throw new HttpException('api-messages:invoice-not-found', HttpStatus.NOT_FOUND);
        }

        return invoice;
    }

    async getStaffBookQuotationById() {
        const book = await this.prisma.book.findMany({
            where: {
                deletedAt: null,
            },
            select: this.prisma.createSelect(['id', 'name', 'price']),
        });

        if (!book) {
            throw new HttpException('api-messages:book-not-found', HttpStatus.NOT_FOUND);
        }

        return book;
    }

    async getInvoiceByBookOrderId(bookOrderId: string) {
        const invoice = await this.prisma.bookOrder.findFirst({
            where: {
                id: bookOrderId,
            },
            select: {
                ...this.prisma.createSelect(['quantity', 'bookPrice', 'displayId', 'createdAt', 'status', 'updatedAt']),
                member: {
                    select: this.prisma.createSelect(['fullName', 'preferredName', 'phoneNumber', 'address']),
                },
                book: {
                    select: this.prisma.createSelect(['name']),
                },
            },
        });

        if (!invoice) {
            throw new HttpException('api-messages:invoice-not-found', HttpStatus.NOT_FOUND);
        }

        return invoice;
    }

    async getMemberBookQuotationById() {
        const book = await this.prisma.book.findMany({
            where: {
                deletedAt: null,
            },
            select: this.prisma.createSelect(['id', 'name', 'price']),
        });

        if (!book) {
            throw new HttpException('api-messages:book-not-found', HttpStatus.NOT_FOUND);
        }

        return book;
    }
}
