import { Injectable, OnModuleDestroy, OnModuleInit } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { Prisma, PrismaClient } from '@prisma/client';
import * as argon from 'argon2';
import * as dayjs from 'dayjs';
import { roleInObject } from 'src/data/role';

type A<T extends string> = T extends `${infer U}ScalarFieldEnum` ? U : never;
type Entity = A<keyof typeof Prisma>;
type Keys<T extends Entity> = Extract<keyof (typeof Prisma)[keyof Pick<typeof Prisma, `${T}ScalarFieldEnum`>], string>;

@Injectable()
export class PrismaService extends PrismaClient implements OnModuleInit, OnModuleDestroy {
    constructor(
        private config: ConfigService,
        // private email: EmailService,
    ) {
        super({
            datasources: {
                db: {
                    url: config.get('DATABASE_URL'),
                },
            },
        });
    }

    async onModuleInit() {
        await this.$connect();
        await this.createDefaultRole();
        await this.createDefaultAccount();
        console.log('DATABASE CONNECTED'.blue.bold);
    }

    async onModuleDestroy() {
        await this.$disconnect();
    }

    private async createDefaultRole() {
        const existingRole = await this.role.findFirst({
            select: {
                id: true,
            },
            where: {
                deletedAt: null,
            },
        });

        if (!existingRole) {
            await this.role.create({
                data: {
                    name: 'Super Admin',
                    superAdmin: true,
                    ...roleInObject(),
                },
            });
            return;
        }

        await this.role.update({
            where: {
                id: existingRole.id,
            },
            data: {
                ...roleInObject(),
            },
        });
    }

    private async createDefaultAccount() {
        const existingStaff = await this.staff.findFirst({
            select: {
                id: true,
            },
            where: {
                email: this.config.get('DEFAULT_STAFF_EMAIL'),
                deletedAt: null,
            },
        });

        if (!existingStaff) {
            // Hash Default Password
            const hashedPassword = await argon.hash(this.config.get('DEFAULT_STAFF_PASSWORD'));

            // Find Super Admin Role Id
            const superAdminRoleId = await this.role.findFirst({
                select: {
                    id: true,
                },
                where: {
                    name: 'Super Admin',
                },
            });

            // Create Default Super Admin
            await this.staff.create({
                data: {
                    email: this.config.get('DEFAULT_STAFF_EMAIL'),
                    fullName: 'Super Admin',
                    password: hashedPassword,
                    phoneNumber: '0123456789',
                    roleId: superAdminRoleId.id,
                    status: 'ACTIVE',
                    lastActiveAt: dayjs().toDate(),
                },
            });
        }
    }

    createSelect = <T extends string>(attributes: T[]): Record<T, true> => {
        const select = {} as Record<T, true>;

        for (const attribute of attributes) {
            select[attribute] = true;
        }

        return select;
    };

    pageCounter = (total: number, page: number, pageSize: number) => {
        const totalPages = Math.ceil(total / pageSize);
        const currentPage = page > totalPages ? 1 : page;

        return currentPage;
    };

    exclude<T extends Entity, K extends Keys<T>>(type: T, omit: K[]) {
        type Key = Exclude<Keys<T>, K>;
        type TMap = Record<Key, true>;
        const result: TMap = {} as TMap;
        for (const key in Prisma[`${type}ScalarFieldEnum`]) {
            if (!omit.includes(key as K)) {
                result[key as Key] = true;
            }
        }
        return result;
    }
}
