import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { Prisma } from '@prisma/client';
import * as dayjs from 'dayjs';
import { MediaService } from 'src/media/media.service';
import { PrismaService } from 'src/prisma/prisma.service';
import { Pagination } from 'src/types';
import { StudyGroup } from 'src/types/study-group';
import { conditionalReturn } from 'src/utils';
import { UploadMedicalCertificateDto } from './study-group.dto';

@Injectable()
export class StudyGroupService {
    constructor(
        private prisma: PrismaService,
        private media: MediaService,
    ) {}

    // Context Provider
    async getStudyGroupById(studyGroupId: string) {
        const studyGroup = await this.prisma.studyGroup.findUnique({
            where: {
                id: studyGroupId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id', 'name', 'startDate', 'endDate', 'status', 'description']),
                book: {
                    select: {
                        ...this.prisma.createSelect(['id', 'name', 'link', 'uploadType']),
                        bookMedias: {
                            select: {
                                media: {
                                    select: {
                                        ...this.prisma.createSelect(['key', 'name']),
                                    },
                                },
                            },
                        },
                    },
                },
                studyGroupTask: {
                    orderBy: {
                        taskDate: 'asc',
                    },
                    select: {
                        ...this.prisma.createSelect(['id', 'taskDate']),
                        studyGroupTaskComment: {
                            where: {
                                deletedAt: null,
                            },
                            select: {
                                ...this.prisma.createSelect(['memberId']),
                            },
                        },
                        penalties: {
                            select: {
                                ...this.prisma.createSelect(['amount']),
                            },
                        },
                    },
                },
            },
        });

        if (!studyGroup) {
            throw new HttpException('api-messages:study-group-not-found', HttpStatus.NOT_FOUND);
        }

        return studyGroup;
    }

    async getStudyGroupMembers(studyGroupId: string) {
        const studyGroupMembers = await this.prisma.member.findMany({
            where: {
                studyGroupMembers: {
                    some: {
                        studyGroupId,
                    },
                },
            },
            select: {
                ...this.prisma.createSelect(['id', 'fullName', 'preferredName', 'phoneNumber']),
                studyGroupMembers: {},
            },
        });

        const memberCount = studyGroupMembers.length;

        return { studyGroupMembers, memberCount };
    }

    async getStudyGroupTaskById(studyGroupTaskId: string) {
        const studyGroupTask = await this.prisma.studyGroupTask.findFirst({
            where: {
                id: studyGroupTaskId,
            },
            select: {
                ...this.prisma.createSelect(['id', 'description', 'taskDate', 'status', 'studyGroupId']),
            },
        });

        if (!studyGroupTask) {
            throw new HttpException('api-messages:study-group-task-not-found', HttpStatus.NOT_FOUND);
        }

        const studyGroup = await this.prisma.studyGroup.findFirst({
            where: {
                id: studyGroupTask.studyGroupId,
            },
        });

        if (!studyGroup) {
            throw new HttpException('api-messages:study-group-not-found', HttpStatus.NOT_FOUND);
        }

        return studyGroupTask;
    }

    ///////////////////
    // Study Group
    ///////////////////

    // Get
    async getAllStudyGroupByPagination(query: StudyGroup, memberId: string) {
        const { page, pageSize, sortField, sortOrder, name, bookName, status, startDate, endDate, createdAt } = query;

        const whereOptions: Prisma.StudyGroupWhereInput = {
            deletedAt: null,
            studyGroupMembers: {
                some: {
                    memberId,
                    deletedAt: null,
                },
            },
            ...conditionalReturn(!!name, {
                name: { mode: 'insensitive', contains: name },
            }),
            ...conditionalReturn(!!bookName, {
                book: {
                    name: { mode: 'insensitive', contains: bookName },
                },
            }),
            ...conditionalReturn(!!status, {
                status,
            }),
            ...conditionalReturn(!!startDate, {
                startDate: {
                    gte: startDate && new Date(dayjs(startDate).startOf('day').toISOString()),
                    lte: startDate && new Date(dayjs(startDate).endOf('day').toISOString()),
                },
            }),
            ...conditionalReturn(!!endDate, {
                endDate: {
                    gte: endDate && new Date(dayjs(endDate).startOf('day').toISOString()),
                    lte: endDate && new Date(dayjs(endDate).endOf('day').toISOString()),
                },
            }),
            ...conditionalReturn(createdAt && createdAt.length === 2, {
                createdAt: {
                    gte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[0]).startOf('day').toISOString()),
                    lte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[1]).endOf('day').toISOString()),
                },
            }),
        };

        const studyGroupCount = await this.prisma.studyGroup.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(studyGroupCount, page, pageSize);

        const studyGroupList = await this.prisma.studyGroup.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: {
                ...this.prisma.createSelect(['id', 'name', 'startDate', 'endDate', 'status', 'createdAt']),
                book: {
                    select: {
                        ...this.prisma.createSelect(['id', 'name']),
                    },
                },
                _count: {
                    select: {
                        studyGroupMembers: true,
                    },
                },
            },
        });

        return {
            count: studyGroupCount,
            rows: studyGroupList,
            page: currentPage,
        };
    }

    async getMemberPenalty(studyGroupId: string, memberId: string) {
        // find study grou task in study group
        const studyGroupTask = await this.prisma.studyGroupTask.findMany({
            where: {
                studyGroupId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        // find member penalties
        const memberPenalty = await this.prisma.penalty.findMany({
            where: {
                studyGroupTaskId: {
                    in: studyGroupTask.map((task) => task.id),
                },
                memberId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id', 'amount', 'createdAt']),
                studyGroupTask: {
                    select: {
                        ...this.prisma.createSelect(['id', 'taskDate']),
                    },
                },
            },
        });

        // calculate total amount of penalties
        const totalAmount = memberPenalty.reduce((acc, curr) => acc + curr.amount, 0);

        return { memberPenalty, totalAmount };
    }

    ///////////////////
    // Study Group Task
    ///////////////////

    // Get
    async getMedicalCertificate(studyGroupTaskId: string, memberId: string) {
        const medicalCertificate = await this.prisma.medicalCertificate.findFirst({
            where: {
                studyGroupTaskId,
                memberId,
            },
            select: {
                ...this.prisma.createSelect(['id', 'applyDate', 'studyGroupTaskId', 'memberId', 'status']),
            },
        });

        return medicalCertificate;
    }

    // Create
    async uploadMedicalCertificateMedia(file: Express.Multer.File) {
        const fileResponse = await this.media.upload(file, { isPublic: false });

        const mediaResponse = await this.prisma.media.create({
            data: fileResponse,
            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
        });

        return mediaResponse;
    }

    async uploadMedicalCertificate(memberId: string, studyGroupTaskId: string, body: UploadMedicalCertificateDto) {
        const member = await this.prisma.member.findUnique({
            where: {
                id: memberId,
            },
            select: {
                id: true,
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        const studyGroupTask = await this.prisma.studyGroupTask.findUnique({
            where: {
                id: studyGroupTaskId,
                deletedAt: null,
            },
            select: this.prisma.createSelect(['taskDate']),
        });

        if (!studyGroupTask) {
            throw new HttpException('api-messages:study-group-task-not-found', HttpStatus.NOT_FOUND);
        }

        const response = await this.prisma.medicalCertificate.create({
            data: {
                applyDate: studyGroupTask.taskDate,
                memberId: member.id,
                studyGroupTaskId: studyGroupTaskId,
                media: {
                    create: body.medias,
                },
            },
            select: this.prisma.createSelect(['id', 'applyDate']),
        });

        return response;
    }

    ///////////////////
    // Commment
    ///////////////////

    // Get
    async getMemberStudyGroupComments(memberId: string, studyGroupId: string) {
        const studyGroupTaskComments = await this.prisma.studyGroupTaskComment.findMany({
            where: {
                memberId,
                studyGroupTask: {
                    studyGroupId,
                },
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id', 'comment', 'commentDate', 'prePlanned', 'prePlannedDate', 'studyGroupTaskId']),
            },
        });

        return studyGroupTaskComments;
    }

    async getTaskComments(studyGroupTaskId: string, memberId: string, query: Pagination) {
        const { page, pageSize } = query;

        const whereOptions: Prisma.StudyGroupTaskCommentWhereInput = {
            studyGroupTaskId,
            deletedAt: null,
            memberId: {
                not: memberId,
            },
            prePlanned: false,
        };

        const studyGroupTaskCommentCount = await this.prisma.studyGroupTaskComment.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(studyGroupTaskCommentCount, page, pageSize);

        const studyGroupTaskComments = await this.prisma.studyGroupTaskComment.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                commentDate: 'desc',
            },
            select: {
                ...this.prisma.createSelect(['id', 'comment', 'commentDate', 'createdAt', 'updatedAt', 'prePlanned', 'prePlannedDate', 'parent_id']),
                member: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName', 'preferredName']),
                    },
                },
                staff: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName']),
                    },
                },
                studyGroupTask: {
                    select: {
                        ...this.prisma.createSelect(['id']),
                    },
                },
                Children: {
                    select: {
                        id: true,
                    }
                }
            },
        });

        return {
            total: studyGroupTaskCommentCount,
            rows: studyGroupTaskComments,
            page: currentPage,
        };
    }

    async getOwnTaskComment(studyGroupTaskId: string, memberId: string) {
        const studyGroupTaskComments = await this.prisma.studyGroupTaskComment.findFirst({
            where: {
                studyGroupTaskId,
                memberId,
                parent_id: null,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id', 'comment', 'commentDate', 'createdAt', 'updatedAt', 'prePlanned', 'prePlannedDate', 'parent_id']),
                member: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName', 'preferredName']),
                    },
                },
                staff: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName']),
                    },
                },
                studyGroupTask: {
                    select: {
                        ...this.prisma.createSelect(['id']),
                    },
                },
                Children: {
                    select: {
                        id: true,
                    }
                }
            },
        });

        return studyGroupTaskComments;
    }

    async getMemberTaskComments(memberId: string, query: StudyGroup) {
        const { name, bookName, status, startDate, endDate, createdAt } = query;

        const whereOptions: Prisma.StudyGroupTaskCommentWhereInput = {
            memberId,
            deletedAt: null,
            prePlanned: false,
            studyGroupTask: {
                studyGroup: {
                    ...conditionalReturn(!!name, {
                        name: { mode: 'insensitive', contains: name },
                    }),
                    ...conditionalReturn(!!bookName, {
                        book: {
                            name: { mode: 'insensitive', contains: bookName },
                        },
                    }),
                    ...conditionalReturn(!!status, {
                        status,
                    }),
                    ...conditionalReturn(!!startDate, {
                        startDate: {
                            gte: startDate && new Date(dayjs(startDate).startOf('day').toISOString()),
                            lte: startDate && new Date(dayjs(startDate).endOf('day').toISOString()),
                        },
                    }),
                    ...conditionalReturn(!!endDate, {
                        endDate: {
                            gte: endDate && new Date(dayjs(endDate).startOf('day').toISOString()),
                            lte: endDate && new Date(dayjs(endDate).endOf('day').toISOString()),
                        },
                    }),
                    ...conditionalReturn(createdAt && createdAt.length === 2, {
                        createdAt: {
                            gte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[0]).startOf('day').toISOString()),
                            lte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[1]).endOf('day').toISOString()),
                        },
                    }),
                },
            },
        };

        const memberTaskComments = await this.prisma.studyGroupTaskComment.findMany({
            where: whereOptions,
            orderBy: {
                studyGroupTask: {
                    taskDate: 'desc',
                },
            },
            select: {
                ...this.prisma.createSelect(['id', 'comment', 'createdAt']),
                studyGroupTask: {
                    select: {
                        ...this.prisma.createSelect(['id', 'taskDate']),
                        studyGroup: {
                            select: {
                                ...this.prisma.createSelect(['id', 'name']),
                            },
                        },
                    },
                },
                member: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName']),
                    },
                },
            },
        });

        return memberTaskComments;
    }

    async getTaskCommentLikes(studyGroupTaskId: string) {
        const studyGroupTaskCommentLikes = await this.prisma.studyGroupTaskCommentLike.findMany({
            where: {
                studyGroupTaskId,
            },
            select: {
                ...this.prisma.createSelect(['id', 'studyGroupTaskId', 'studyGroupTaskCommentId', 'memberId', 'staffId']),
            },
        });

        return studyGroupTaskCommentLikes;
    }

    private async getNestedReplies(studyGroupTaskId: string, parentId: string) {
        const replies = await this.prisma.studyGroupTaskComment.findMany({
            where: {
                studyGroupTaskId,
                parent_id: parentId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id', 'comment', 'studyGroupTaskId', 'commentDate', 'prePlanned', 'prePlannedDate', 'parent_id']),
                member: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName', 'preferredName']),
                    },
                },
                staff: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName']),
                    },
                },
                Children: {
                    select: {
                        id: true,
                    }
                }
            },
        });

        // Recursively fetch nested replies
        const nestedReplies = await Promise.all(
            replies.map(async (reply) => {
                const children = await this.getNestedReplies(studyGroupTaskId, reply.id);
                return {
                    ...reply,
                    Children: children,
                };
            }),
        );

        return nestedReplies;
    }

    async getReplies(studyGroupTaskId: string, parentId: string) {
        const studyGroupTaskCommentReplies = await this.getNestedReplies(studyGroupTaskId, parentId);
        return studyGroupTaskCommentReplies;
    }

    // Update
    async updateComment(studyGroupTaskId: string, comment: string, memberId: string) {
        // Check if the study group task exists
        const studyGroupTask = await this.prisma.studyGroupTask.findFirst({
            where: {
                id: studyGroupTaskId,
                deletedAt: null,
            },
        });

        if (!studyGroupTask) {
            throw new HttpException('api-messages:study-group-task-not-found', HttpStatus.NOT_FOUND);
        }

        const studyGroupTaskComment = await this.prisma.studyGroupTaskComment.findFirst({
            where: {
                studyGroupTaskId,
                memberId,
                deletedAt: null,
            },
        });

        if (!studyGroupTaskComment) {
            const createdComment = await this.prisma.studyGroupTaskComment.create({
                data: {
                    comment,
                    memberId,
                    studyGroupTaskId: studyGroupTask.id,
                    commentDate: new Date(),
                },
            });

            return createdComment;
        }

        // Update the existing comment
        const updatedComment = await this.prisma.studyGroupTaskComment.update({
            where: {
                id: studyGroupTaskComment.id,
                deletedAt: null,
            },
            data: {
                commentDate: new Date(),
                prePlanned: false,
                comment,
            },
        });

        return updatedComment;
    }

    async updateReply(studyGroupTaskId: string, memberId: string, commentId: string, reply: string) {
        const studyGroupTaskComment = await this.prisma.studyGroupTaskComment.findFirst({
            where: {
                id: commentId,
                studyGroupTaskId,
                deletedAt: null,
            },
        });

        if (!studyGroupTaskComment) {
            throw new HttpException('api-messages:study-group-task-comment-not-found', HttpStatus.NOT_FOUND);
        }

        const memberStudyGroupTaskReply = await this.prisma.studyGroupTaskComment.findFirst({
            where: {
                memberId,
                studyGroupTaskId,
                parent_id: commentId,
                deletedAt: null,
            },
        });

        if (memberStudyGroupTaskReply) {
            const updatedReply = await this.prisma.studyGroupTaskComment.update({
                where: {
                    id: memberStudyGroupTaskReply.id,
                    deletedAt: null,
                },
                data: {
                    comment: reply,
                    commentDate: new Date(),
                },
            });

            return updatedReply;
        } else {
            const createdReply = await this.prisma.studyGroupTaskComment.create({
                data: {
                    comment: reply,
                    memberId,
                    studyGroupTaskId,
                    commentDate: new Date(),
                    parent_id: commentId,
                },
            });

            return createdReply;
        }
    }

    async scheduleComment(studyGroupTaskId: string, comment: string, prePlannedDate: string, memberId: string) {
        const studyGroupTask = await this.prisma.studyGroupTask.findFirst({
            where: {
                id: studyGroupTaskId,
                deletedAt: null,
            },
        });

        if (!studyGroupTask) {
            throw new HttpException('api-messages:study-group-task-not-found', HttpStatus.NOT_FOUND);
        }

        const studyGroupTaskComment = await this.prisma.studyGroupTaskComment.findFirst({
            where: {
                studyGroupTaskId,
                memberId,
                deletedAt: null,
            },
        });

        if (!studyGroupTaskComment) {
            await this.prisma.studyGroupTaskComment.create({
                data: {
                    comment,
                    memberId,
                    prePlanned: true,
                    prePlannedDate,
                    studyGroupTaskId: studyGroupTask.id,
                },
            });

            return;
        }

        const updatedScheduleComment = await this.prisma.studyGroupTaskComment.update({
            where: {
                id: studyGroupTaskComment.id,
                deletedAt: null,
            },
            data: {
                comment,
            },
        });

        return updatedScheduleComment;
    }

    async likeTaskComment(studyGroupTaskCommentId: string, memberId: string, studyGroupTaskId: string) {
        const studyGroupTaskComment = await this.prisma.studyGroupTaskComment.findFirst({
            where: {
                id: studyGroupTaskCommentId,
                deletedAt: null,
            },
        });

        if (!studyGroupTaskComment) {
            throw new HttpException('api-messages:study-group-task-comment-not-found', HttpStatus.NOT_FOUND);
        }

        let studyGroupTaskCommentLike = await this.prisma.studyGroupTaskCommentLike.findFirst({
            where: {
                studyGroupTaskCommentId,
                memberId,
            },
        });

        if (studyGroupTaskCommentLike) {
            await this.prisma.studyGroupTaskCommentLike.delete({
                where: {
                    id: studyGroupTaskCommentLike.id,
                },
            });
            studyGroupTaskCommentLike = null;
        } else {
            studyGroupTaskCommentLike = await this.prisma.studyGroupTaskCommentLike.create({
                data: {
                    memberId,
                    studyGroupTaskCommentId,
                    studyGroupTaskId,
                },
            });
        }

        return studyGroupTaskCommentLike;
    }
}
