import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { BookStatus, Prisma, StudyBookRequestStatus, StudyGroupStatus } from '@prisma/client';
import { PrismaService } from 'src/prisma/prisma.service';
import { BookQuery } from 'src/staff/books/book.dto';
import { conditionalReturn } from 'src/utils';

@Injectable()
export class BookService {
    constructor(private prisma: PrismaService) {}

    async getBookByPagination(query: BookQuery) {
        const { page, pageSize, sortField, sortOrder, name } = query;

        /* Where Options */
        const whereOptions: Prisma.BookWhereInput = {
            deletedAt: null,
            status: BookStatus.ACTIVE,
            ...conditionalReturn(!!name, {
                name: { mode: 'insensitive', contains: name },
            }),
        };

        const bookCount = await this.prisma.book.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(bookCount, page, pageSize);

        const bookList = await this.prisma.book.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: {
                ...this.prisma.createSelect(['id', 'name', 'description', 'createdAt', 'availableForBuy', 'price']),
                studyBookRequest: {
                    select: {
                        status: true,
                    },
                },
                bookImages: {
                    select: {
                        media: {
                            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
                        },
                    },
                },
            },
        });

        return {
            count: bookCount,
            rows: bookList,
            page: currentPage,
        };
    }

    async getBookById(bookId: string) {
        const book = await this.prisma.book.findFirst({
            where: {
                id: bookId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.exclude('Book', ['deletedAt', 'updatedAt']),
                bookMedias: {
                    select: {
                        media: {
                            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
                        },
                    },
                },
                bookImages: {
                    select: {
                        mediaId: true,
                        media: {
                            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
                        },
                    },
                },
            },
        });

        if (!book) {
            throw new HttpException('api-messages:book-not-found', HttpStatus.NOT_FOUND);
        }

        return book;
    }

    async getBookRequestByPagination(query: BookQuery, memberId: string) {
        const { page, pageSize, sortField, sortOrder, name } = query;

        /* Where Options */
        const whereOptions: Prisma.StudyBookRequestWhereInput = {
            deletedAt: null,
            memberId,
            ...conditionalReturn(!!name, {
                book: {
                    name: { mode: 'insensitive', contains: name },
                },
            }),
        };

        const bookRequestCount = await this.prisma.studyBookRequest.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(bookRequestCount, page, pageSize);

        const bookRequestList = await this.prisma.studyBookRequest.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: {
                ...this.prisma.createSelect(['id', 'bookId', 'createdAt', 'status', 'reason']),
                book: {
                    select: {
                        ...this.prisma.createSelect(['name', 'description']),
                    },
                },
            },
        });

        return {
            count: bookRequestCount,
            rows: bookRequestList,
            page: currentPage,
        };
    }

    async requestBook(bookId: string, memberId: string) {
        const book = await this.prisma.book.findUnique({
            where: {
                id: bookId,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!book) {
            throw new HttpException('api-messages:book-not-found', HttpStatus.NOT_FOUND);
        }

        // Check if member has enough book tokens
        const member = await this.prisma.member.findUnique({
            where: {
                id: memberId,
            },
            select: {
                bookTokens: true,
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        if (member.bookTokens < 1) {
            throw new HttpException('api-messages:you-dont-have-enough-book-tokens', HttpStatus.BAD_REQUEST);
        }

        // Check if member has a pending request for the same book
        const existingRequest = await this.prisma.studyBookRequest.findFirst({
            where: {
                memberId,
                bookId,
                status: StudyBookRequestStatus.PENDING,
            },
        });

        if (existingRequest) {
            throw new HttpException('api-messages:you-have-a-request-that-is-currently-pending', HttpStatus.CONFLICT);
        }

        const existingGroupMember = await this.prisma.studyGroupMember.findFirst({
            where: {
                memberId,
                studyGroup: {
                    status: {
                        in: [StudyGroupStatus.ONGOING],
                    },
                    bookId,
                },
            },
            select: {
                studyGroup: true,
            },
        });

        if (existingGroupMember) {
            throw new HttpException('api-messages:you-have-an-ongoing/active-study-group-with-this-book', HttpStatus.CONFLICT);
        }

        // Transaction, deduct book token and create request
        const response = await this.prisma.$transaction([
            this.prisma.member.update({
                where: {
                    id: memberId,
                },
                data: {
                    bookTokens: {
                        decrement: 1,
                    },
                },
            }),
            this.prisma.studyBookRequest.create({
                data: {
                    bookId,
                    memberId,
                    status: StudyBookRequestStatus.PENDING,
                },
            }),
        ]);

        return response;
    }

    async cancelBookRequest(bookId: string, requestId: string, memberId: string) {
        const request = await this.prisma.studyBookRequest.findFirst({
            where: {
                id: requestId,
                memberId,
                bookId,
                status: StudyBookRequestStatus.PENDING,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!request) {
            throw new HttpException('api-messages:book-request-not-found', HttpStatus.NOT_FOUND);
        }

        // Transaction, cancel request and return book token
        const response = await this.prisma.$transaction([
            this.prisma.member.update({
                where: {
                    id: memberId,
                },
                data: {
                    bookTokens: {
                        increment: 1,
                    },
                },
            }),
            this.prisma.studyBookRequest.update({
                where: {
                    id: requestId,
                },
                data: {
                    status: StudyBookRequestStatus.CANCELLED,
                },
            }),
        ]);

        return response;
    }
}
