import { Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { DeleteObjectCommand, GetObjectCommand, HeadObjectCommand, PutObjectCommand, S3Client } from '@aws-sdk/client-s3';
import * as dayjs from 'dayjs';
import { Media } from 'src/types';
import { v4 as uuidv4 } from 'uuid';

@Injectable()
export class S3Service {
    private s3: S3Client;
    private bucketName: string;
    private environment: string;
    constructor(private config: ConfigService) {
        this.s3 = new S3Client({
            region: config.get('AWS_BUCKET_REGION'),
            credentials: {
                accessKeyId: config.get('AWS_ACCESS_KEY'),
                secretAccessKey: config.get('AWS_SECRET_KEY'),
            },
        });

        this.bucketName = config.get('AWS_BUCKET_NAME');
        this.environment = config.get<string>('NODE_ENV') === 'production' ? 'production' : 'development';
    }

    keyGenerator(filename: string) {
        const extension = filename.split('.').pop();
        const key = `${dayjs().valueOf()}-${uuidv4()}.${extension}`;

        return key;
    }

    async uploadFile(media: Media) {
        const key = this.keyGenerator(media.originalname);
        const keyWithPrefix = this.environment + '/' + key;

        const uploadParams = {
            Bucket: this.bucketName as string,
            Body: media.buffer,
            Key: keyWithPrefix,
        };

        const putCommand = new PutObjectCommand(uploadParams);

        await this.s3.send(putCommand);

        return key;
    }

    async isObjectExist(key: string) {
        try {
            const keyWithPrefix = this.environment + '/' + key;
            const headObjectCommand = new HeadObjectCommand({
                Bucket: this.bucketName,
                Key: keyWithPrefix,
            });
            await this.s3.send(headObjectCommand);

            return true;
        } catch (error) {
            return false;
        }
    }

    async getFileStream(key: string) {
        const keyWithPrefix = this.environment + '/' + key;
        const downloadParams = {
            Key: keyWithPrefix,
            Bucket: this.bucketName,
        };

        const getCommand = new GetObjectCommand(downloadParams);
        const result = await this.s3.send(getCommand);

        if (!result.Body) {
            return false;
        }

        return result;
    }

    async deleteFile(key: string) {
        const keyWithPrefix = this.environment + '/' + key;
        const deleteParams = {
            Key: keyWithPrefix,
            Bucket: this.bucketName,
        };

        const deleteCommand = new DeleteObjectCommand(deleteParams);
        await this.s3.send(deleteCommand);

        return true;
    }
}
