import { CanActivate, ExecutionContext, HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { Reflector } from '@nestjs/core';
import { JwtService } from '@nestjs/jwt';
import { Request } from 'express';
import { USER_TOKEN_COOKIE_NAME } from 'src/config';
import roleData from 'src/data/role';
import { PrismaService } from 'src/prisma/prisma.service';
import {
    MemberInRequest,
    MemberJwtPayload,
    PERMISSION,
    RequestWithMember,
    RequestWithStaff,
    StaffInRequest,
    StaffJwtPayload,
    Token,
    User,
} from 'src/types';
import { CustomException } from 'src/utils/CustomException';

@Injectable()
export class AuthenticationGuard implements CanActivate {
    constructor(
        private reflector: Reflector,
        private config: ConfigService,
        private jwtService: JwtService,
        private prisma: PrismaService,
    ) {}

    async canActivate(context: ExecutionContext): Promise<boolean> {
        const authenticator = this.reflector.get<User>('authenticator', context.getHandler());
        const request = context.switchToHttp().getRequest<Request>();

        switch (authenticator) {
            case User.STAFF: {
                const staffToken = request.cookies[USER_TOKEN_COOKIE_NAME.staff] as Token;
                const verifiedStaff = await this.staffAuthenticator(staffToken);
                (request as unknown as RequestWithStaff).staff = verifiedStaff;
                break;
            }
            case User.MEMBER: {
                const memberToken = request.cookies[USER_TOKEN_COOKIE_NAME.member] as Token;
                const verifiedMember = await this.memberAuthenticator(memberToken);
                (request as unknown as RequestWithMember).member = verifiedMember;
                break;
            }
            default: {
                break;
            }
        }

        return true;
    }

    private async staffAuthenticator(token: Token) {
        if (!token) {
            throw new HttpException('api-messages:unauthorized', HttpStatus.UNAUTHORIZED);
        }
        // Implement your own staff authentication logic here.
        const decoded: StaffJwtPayload = await this.unsignToken(token);

        if (!decoded) {
            throw new HttpException('api-messages:unauthorized', HttpStatus.UNAUTHORIZED);
        }

        const staff = await this.prisma.staff.findFirst({
            where: {
                id: decoded.id,
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
                password: true,
                status: true,
                role: {
                    select: this.prisma.createSelect(roleData.exclude(['id', 'name', 'deletedAt', 'createdAt', 'updatedAt', 'superAdmin'])),
                },
            },
        });

        // Check if staff is exist or not
        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        // Check if the password is null
        if (!staff.password) {
            throw new HttpException('api-messages:pending-for-verification', HttpStatus.UNAUTHORIZED);
        }

        if (staff.status !== 'ACTIVE') {
            throw new HttpException('api-messages:staff-not-active', HttpStatus.UNAUTHORIZED);
        }

        delete staff.password;

        return staff as unknown as StaffInRequest;
    }

    private async memberAuthenticator(token: Token) {
        if (!token) {
            throw new HttpException('api-messages:unauthorized', HttpStatus.UNAUTHORIZED);
        }
        // Implement your own member authentication logic here.
        const decoded: MemberJwtPayload = await this.unsignToken(token);

        if (!decoded) {
            throw new HttpException('api-messages:unauthorized', HttpStatus.UNAUTHORIZED);
        }

        const member = await this.prisma.member.findFirst({
            where: {
                id: decoded.id,
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
                password: true,
                status: true,
            },
        });

        // Check if member exist or not
        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        // Check if the password is null
        if (!member.password) {
            throw new HttpException('api-messages:pending-for-verification', HttpStatus.UNAUTHORIZED);
        }

        // TODO: check status
        if (member.status !== 'ACTIVE') {
            throw new HttpException('api-messages:member-not-active', HttpStatus.UNAUTHORIZED);
        }

        delete member.password;

        return member as unknown as MemberInRequest;
    }

    private async unsignToken(token: Token) {
        try {
            const decoded: StaffJwtPayload = this.jwtService.verify(token, {
                secret: this.config.get('TOKEN_SECRET'),
            });
            return decoded;
        } catch (error) {
            throw new HttpException('api-messages:unauthorized', HttpStatus.UNAUTHORIZED);
        }
    }
}

@Injectable()
export class PermissionsGuard implements CanActivate {
    constructor(private reflector: Reflector) {}

    canActivate(context: ExecutionContext): boolean {
        const permissionToAuthorize = this.reflector.get<PERMISSION>('authorizer', context.getHandler());
        if (!permissionToAuthorize) {
            return true;
        }

        const authenticator = this.reflector.get<User>('authenticator', context.getHandler());
        if (authenticator !== User.STAFF) {
            throw new Error('Only staff access can use permission decorator');
        }

        const request = context.switchToHttp().getRequest() as RequestWithStaff;
        const staff = request.staff;

        if (!staff || !staff.role[permissionToAuthorize]) {
            throw new CustomException('api-messages:unauthorized', HttpStatus.UNAUTHORIZED, {
                unauthorized: true,
            });
        }

        return true;
    }
}
