import { GetServerSideProps, NextPage } from 'next';
import { useTranslation } from 'next-i18next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import { Tabs } from 'antd';
import Layout from '@/components/layouts';
import { BasePageProps } from '@/types';
import { authentication } from '@/utils/authentication';
import localeLinkGenerator from '@/utils/localeLinkGenerator';
import PenaltySettingTab from '@/components/site-setting/tabs/Penalty';
import BankDetailsSettingTab from '@/components/site-setting/tabs/CompanyInformation';
import BookTokenTab from '@/components/site-setting/tabs/BookToken';
import Notice from '@/components/site-setting/tabs/Notice';

const SiteSettingPage: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['site-setting', 'layout', 'common', 'messages']);

    // Tab Items
    const tabItems = [
        // Book Token Tab
        {
            key: 'book-token',
            label: t('book-token'),
            children: <BookTokenTab />,
        },
        // Penalty Tab
        {
            key: 'penalty',
            label: t('penalty'),
            children: <PenaltySettingTab />,
        },
        {
            key: 'company-information',
            label: t('company-information'),
            children: <BankDetailsSettingTab />,
        },
        {
            key: 'notice',
            label: t('notice'),
            children: <Notice />,
        },
    ];

    const seoConfig = {
        title: t('site-setting'),
    };

    const breadCrumbItems = [
        {
            label: t('layout:site-setting'),
            path: '/',
        },
    ];

    return (
        <Layout staff={staff} breadCrumbItems={breadCrumbItems} activeMenu={['site-setting']} seoConfig={seoConfig} withBackground>
            <Tabs items={tabItems} />
        </Layout>
    );
};

export default SiteSettingPage;

export const getServerSideProps: GetServerSideProps = async ({ req, locale, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req, 'SITE_SETTING');

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['site-setting', 'layout', 'common', 'messages', 'api-messages'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }
        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};
