import { toast } from 'react-toastify';
import { GetServerSideProps, NextPage } from 'next';
import { Trans, useTranslation } from 'next-i18next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import Link from 'next/link';
import { PlusCircleOutlined } from '@ant-design/icons';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { Button, MenuProps, Space, Table } from 'antd';
import { ColumnsType } from 'antd/es/table';
import ConfirmationModal from '@/components/modals/ConfirmationModal';
import ActionDropdown from '@/components/ui/ActionDropdown';
import usePagination from '@/hooks/usePagination';
import Layout from '@/components/layouts';
import { deleteRole, getRoleListByPagination, restoreRole } from '@/services/role';
import { AxiosErrorResponse, BasePageProps, Role } from '@/types';
import { authentication } from '@/utils/authentication';
import { errorMessageFormatter } from '@/utils/errorFormatter';
import localeLinkGenerator from '@/utils/localeLinkGenerator';
import { dateTimeTransformer } from '@/utils/timeTransformer';

const RolesPage: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['role', 'messages', 'common']);
    const [pagination, setPagination, paginationOnChange] = usePagination<Role>();
    const queryClient = useQueryClient();

    const breadCrumbItems = [
        {
            label: t('role'),
            path: '/role',
        },
    ];

    const seoConfig = {
        title: t('role'),
    };

    const rolesColumn: ColumnsType<Role> = [
        {
            title: t('name'),
            dataIndex: 'name',
            render: (_: string, record: Role) => (
                <Link href={`/role/${record.id}`} className="font-bold">
                    {record.name}
                </Link>
            ),
        },
        {
            title: t('common:created-at'),
            dataIndex: 'createdAt',
            sorter: true,
            width: 200,
            render: (createdAt: string) => {
                return dateTimeTransformer(createdAt);
            },
        },
        {
            title: t('common:updated-at'),
            dataIndex: 'updatedAt',
            sorter: true,
            width: 200,
            render: (updatedAt: string) => {
                return dateTimeTransformer(updatedAt);
            },
        },
        {
            width: 64,
            fixed: 'right',
            key: 'actions',
            render: (_: string, record: Role) => {
                const items: MenuProps['items'] = [
                    // View Role
                    {
                        label: <Link href={`/role/${record.id}`}>{t('common:view')}</Link>,
                        key: 'view',
                        className: 'text-center',
                    },
                    // Delete Role
                    {
                        label: (
                            <ConfirmationModal
                                title={t('common:delete-confirmation')}
                                message={
                                    <Trans
                                        i18nKey={'messages:are-you-sure-you-want-to-delete'}
                                        components={{ strong: <strong /> }}
                                        values={{ name: record.name }}
                                    />
                                }
                                okText={t('common:delete')}
                                onOk={() => {
                                    onDeleteRoleHandler(record.id);
                                }}
                                reason={false}
                                okButtonProps={{
                                    danger: true,
                                }}
                                width={400}
                            >
                                <Button type="text" block className="!p-0 hover:!text-white" style={{ transition: 'none' }}>
                                    {t('common:delete')}
                                </Button>
                            </ConfirmationModal>
                        ),
                        key: 'delete',
                        danger: true,
                        className: '!p-0',
                    },
                ];

                return (
                    <Space>
                        <ActionDropdown items={items} />
                    </Space>
                );
            },
        },
    ];

    // Roles Query
    const rolesQuery = useQuery({
        queryKey: ['role', 'pagination', pagination], // When pagination changed, it will trigger a new query
        keepPreviousData: true,
        queryFn: async () => {
            const query = {
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const res = await getRoleListByPagination(query);

            setPagination((prevValue) => {
                return {
                    ...prevValue,
                    page: res.data.page,
                    total: res.data.total,
                };
            });

            return res.data.rows;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    const deleteRoleMutation = useMutation({
        mutationFn: async (roleId: string) => {
            const res = await deleteRole(roleId);
            return res.data;
        },
        onSuccess: () => {
            queryClient.invalidateQueries({
                queryKey: ['role', 'pagination'],
            });
        },
    });

    const restoreRoleMutation = useMutation({
        mutationFn: async (roleId: string) => {
            const res = await restoreRole(roleId);
            return res.data;
        },
        onSuccess: () => {
            queryClient.invalidateQueries({
                queryKey: ['role', 'pagination'],
            });
        },
    });

    const onDeleteRoleHandler = (id: string) => {
        toast.promise(deleteRoleMutation.mutateAsync(id), {
            pending: {
                render() {
                    return t('messages:deleting-role');
                },
            },
            success: {
                render() {
                    return (
                        <div className="flex items-center justify-between">
                            <p className="p-0 m-0">{t('messages:role-deleted')}</p>
                            <Button
                                onClick={(e) => {
                                    e.stopPropagation();
                                    e.preventDefault();
                                    toast.dismiss();
                                    toast.promise(restoreRoleMutation.mutateAsync(id), {
                                        pending: {
                                            render() {
                                                return t('messages:restoring-role');
                                            },
                                        },
                                        success: {
                                            render() {
                                                return t('messages:role-restored');
                                            },
                                        },
                                        error: {
                                            render({ data }: any) {
                                                return t(errorMessageFormatter(data));
                                            },
                                        },
                                    });
                                }}
                                size="small"
                                className="ml-2"
                            >
                                {t('common:undo')}
                            </Button>
                        </div>
                    );
                },
            },
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    return (
        <Layout staff={staff} breadCrumbItems={breadCrumbItems} activeMenu={['role']} seoConfig={seoConfig} withBackground>
            {/* Header */}
            <div className="flex flex-row justify-end">
                {staff.role.ROLE_CREATE && (
                    <Link href="/role/add">
                        <Button type="primary" icon={<PlusCircleOutlined />}>
                            {t('add-role')}
                        </Button>
                    </Link>
                )}
            </div>
            {/* Table */}
            <div className="mt-5">
                <Table
                    loading={rolesQuery.isFetching}
                    columns={rolesColumn}
                    dataSource={rolesQuery.data}
                    rowKey={(record) => record.id}
                    scroll={{ x: 800 }}
                    onChange={paginationOnChange}
                    pagination={{
                        current: pagination.page,
                        pageSize: pagination.pageSize,
                        defaultPageSize: 1,
                        showSizeChanger: true,
                        pageSizeOptions: [5, 10, 25, 50, 100],
                        showTotal: (total, range) => t('common:pagination', { range0: range[0], range1: range[1], total: total }),
                        total: pagination.total,
                    }}
                />
            </div>
        </Layout>
    );
};

export default RolesPage;

export const getServerSideProps: GetServerSideProps = async ({ locale, req, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req, 'ROLE_VIEW');

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['role', 'layout', 'common', 'messages', 'api-messages'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};
