import Layout from '@/components/layouts';
import ConfirmationModal from '@/components/modals/ConfirmationModal';
import usePagination from '@/hooks/usePagination';
import { approveMedicalCertificate, getMedicalCertificationList, rejectMedicalCertificate } from '@/services/medical-certificate';
import { AxiosErrorResponse, BasePageProps, MedicalCertificate } from '@/types';
import { authentication, errorMessageFormatter, localeLinkGenerator, mediaUrlGenerator } from '@/utils';
import { dateTimeTransformer } from '@/utils/timeTransformer';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { Button, Table, Tag } from 'antd';
import { ColumnsType } from 'antd/es/table';
import { GetServerSideProps, NextPage } from 'next';
import { useTranslation } from 'next-i18next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import Link from 'next/link';
import { toast } from 'react-toastify';

const MedicalCertificatePage: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['medical-certificate', 'common', 'messages', 'layout']);
    const [pagination, setPagination, paginationOnChange] = usePagination<MedicalCertificate>();
    const queryClient = useQueryClient();

    const { isFetching, data, refetch } = useQuery({
        queryKey: ['medical-certificate'],
        queryFn: async () => {
            const query = {
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const response = await getMedicalCertificationList(query);

            setPagination((prev) => {
                return {
                    ...prev,
                    page: response.data.page,
                    total: response.data.total,
                };
            });

            return response.data.rows;
        },
    });

    // Mutations
    const approveMedicalCertificateMutation = useMutation({
        mutationFn: async (id: string) => {
            const response = await approveMedicalCertificate(id);
            return response.data;
        },
        onSuccess() {
            refetch();
            queryClient.invalidateQueries(['medical-certificate', 'all']);
        },
    });

    const rejectMedicalCertificateMutation = useMutation({
        mutationFn: async ({ id, reason }: { id: string; reason: string }) => {
            const response = await rejectMedicalCertificate(id, reason);
            return response.data;
        },
        onSuccess() {
            refetch();
            queryClient.invalidateQueries(['medical-certificate', 'all']);
        },
    });

    // Functions
    const onApproveMedicalCertificateHandler = (id: string) => {
        toast.promise(approveMedicalCertificateMutation.mutateAsync(id), {
            pending: t('messages:approving-medical-certificate'),
            success: t('messages:medical-certificate-approved'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    const onRejectMedicalCertificateHandler = (id: string, reason: string) => {
        toast.promise(rejectMedicalCertificateMutation.mutateAsync({ id, reason }), {
            pending: t('messages:rejecting-medical-certificate'),
            success: t('messages:medical-certificate-rejected'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    const columns: ColumnsType<MedicalCertificate> = [
        {
            title: t('member'),
            dataIndex: ['member', 'fullName'],
            key: 'displayId',
            render: (fullName: string) => {
                return (
                    <Button type="link" className="p-0 font-bold">
                        {fullName}
                    </Button>
                );
            },
        },
        {
            title: t('study-group'),
            dataIndex: ['studyGroupTask', 'studyGroup', 'name'],
            key: 'study-group',
            render: (name: string) => name,
        },
        {
            title: t('apply-date'),
            dataIndex: 'applyDate',
            key: 'apply-date',
            render: (applyDate: string) => {
                return dateTimeTransformer(applyDate);
            },
        },
        {
            title: t('file'),
            dataIndex: ['media', 0, 'media'],
            key: 'media',
            render: (media: { name: string; key: string }) => {
                return (
                    <Link target="_blank" className="font-bold" href={mediaUrlGenerator(media.key)}>
                        {media.name}
                    </Link>
                );
            },
        },
        {
            title: t('status'),
            dataIndex: 'status',
            key: 'status',
            render: (status: string) => {
                switch (status) {
                    case 'APPROVED':
                        return <Tag color="green">{t('common:approved')}</Tag>;
                    case 'REJECTED':
                        return <Tag color="red">{t('common:rejected')}</Tag>;
                    default:
                        return <Tag color="yellow">{t('common:pending')}</Tag>;
                }
            },
        },
        {
            title: t('common:actions'),
            dataIndex: 'id',
            key: 'action',
            render: (_: unknown, record) => {
                return (
                    <>
                        {record.status === 'PENDING' ? (
                            <>
                                <Button
                                    loading={approveMedicalCertificateMutation.isLoading}
                                    type="primary"
                                    className="mr-2"
                                    onClick={() => {
                                        onApproveMedicalCertificateHandler(record.id);
                                    }}
                                >
                                    {t('approve')}
                                </Button>
                                <ConfirmationModal
                                    title={t('common:reject-confirmation')}
                                    message={t('are-you-sure-you-want-to-reject-this-medical-certificate?')}
                                    okText={t('common:reject')}
                                    onOk={(reason) => {
                                        onRejectMedicalCertificateHandler(record.id, reason as string);
                                    }}
                                    reason={true}
                                    okButtonProps={{
                                        danger: true,
                                    }}
                                >
                                    <Button loading={rejectMedicalCertificateMutation.isLoading} danger>
                                        {t('reject')}
                                    </Button>
                                </ConfirmationModal>
                            </>
                        ) : (
                            '-'
                        )}
                    </>
                );
            },
        },
    ];

    const breadCrumbItems = [
        {
            label: t('layout:medical-certificate'),
            path: '/medical-certificate',
        },
    ];

    return (
        <Layout
            staff={staff}
            breadCrumbItems={breadCrumbItems}
            seoConfig={{
                title: t('medical-certificate'),
            }}
            withBackground
            activeMenu={['medical-certificate']}
        >
            <div>
                <div className="table_container mt-4">
                    <Table
                        columns={columns}
                        dataSource={data}
                        loading={isFetching}
                        rowKey={(record) => record.id}
                        scroll={{ x: 1000 }}
                        onChange={paginationOnChange}
                        pagination={{
                            current: pagination.page,
                            pageSize: pagination.pageSize,
                            defaultPageSize: 1,
                            showSizeChanger: true,
                            pageSizeOptions: [10, 25, 50, 100],
                            showTotal: (total, range) => t('common:pagination', { range0: range[0], range1: range[1], total: total }),
                            total: pagination.total,
                        }}
                    />
                </div>
            </div>
        </Layout>
    );
};

export default MedicalCertificatePage;

export const getServerSideProps: GetServerSideProps = async ({ locale, req, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req);

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['medical-certificate', 'layout', 'common', 'messages', 'auth', 'api-messages'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};
