import { GetServerSideProps, NextPage } from 'next';
import Layout from '@/components/layouts';
import { useTranslation } from 'next-i18next';
import { authentication, conditionalReturn, currencyFormatter, errorMessageFormatter, localeLinkGenerator } from '@/utils';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import { AxiosErrorResponse, BasePageProps, Book, BookStatus } from '@/types';
import { Button, Form, MenuProps, Space, Table, Tag } from 'antd';
import { PlusCircleOutlined } from '@ant-design/icons';
import Link from 'next/link';
import { useQuery } from '@tanstack/react-query';
import { bookKeys, getBookList } from '@/services/book';
import usePagination from '@/hooks/usePagination';
import { toast } from 'react-toastify';
import { ColumnsType } from 'antd/es/table';
import AddBook from '@/components/book/modals/AddBook';
import { useState } from 'react';
import FilterDrawer from '@/components/book/modals/Filter';
import ActionDropdown from '@/components/ui/ActionDropdown';
import DeleteBookAction from '@/components/book/DeleteBookAction';
import HtmlParser from '@/components/HtmlParser';
import { dateTimeTransformer } from '@/utils/timeTransformer';

const Index: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['book', 'layout', 'common', 'messages']);
    const [pagination, setPagination, paginationOnChange] = usePagination<Book>();
    const [filterBookForm] = Form.useForm();

    const [isAddBookModalOpen, setIsAddBookModalOpen] = useState<boolean>(false);

    const bookListQuery = useQuery({
        queryKey: bookKeys.pagination(pagination),
        queryFn: async () => {
            const filterValues = filterBookForm.getFieldsValue();
            const query = {
                ...filterValues,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };
            const response = await getBookList(query);

            setPagination((prev) => {
                return {
                    ...prev,
                    page: response.data?.page,
                    total: response.data?.total,
                };
            });

            return response.data?.rows;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    const columns: ColumnsType<Book> = [
        {
            title: t('book-name'),
            dataIndex: 'name',
            key: 'name',
            render: (name: string, book) => (
                <Link className="font-bold" href={`/book/${book.id}`}>
                    {name}
                </Link>
            ),
        },
        {
            title: t('description'),
            dataIndex: 'description',
            key: 'description',
            render: (description: string) => <HtmlParser html={description ?? '-'} />,
        },
        {
            title: t('availableForPurchase'),
            dataIndex: 'availableForBuy',
            key: 'availableForBuy',
            render: (availableForBuy: boolean) => (availableForBuy ? <Tag color="green">{t('yes-purchase')}</Tag> : <Tag>{t('no-purchase')}</Tag>),
        },
        {
            title: t('price'),
            dataIndex: 'price',
            key: 'price',
            render: (price: number) => `RM ${currencyFormatter(price)}`,
        },
        {
            dataIndex: 'status',
            title: t('status'),
            sorter: true,
            render: (status: BookStatus) => {
                return status === BookStatus.ACTIVE ? <Tag color="green">{t('active')}</Tag> : <Tag color="red">{t('inactive')}</Tag>;
            },
        },
        {
            title: t('common:created-at'),
            dataIndex: 'createdAt',
            key: 'createdAt',
            width: 150,
            sorter: true,
            render: (createdAt: string) => dateTimeTransformer(createdAt),
        },
        {
            title: t('common:actions'),
            dataIndex: 'id',
            key: 'id',
            width: 100,
            align: 'center',
            render: (_: unknown, book) => {
                const items: MenuProps['items'] = [
                    {
                        label: <Link href={`/book/${book.id}`}>{t('common:view')}</Link>,
                        key: 'view',
                        className: 'text-center',
                    },
                    ...conditionalReturn(!!staff.role.BOOK_DELETE, [
                        {
                            label: <DeleteBookAction book={book} isTable />,
                            key: 'delete',
                            danger: true,
                            className: 'text-center',
                        },
                    ]),
                ];
                return (
                    <Space>
                        <ActionDropdown items={items} />
                    </Space>
                );
            },
        },
    ];

    const onResetFilterHandler = () => {
        filterBookForm.resetFields();
        bookListQuery.refetch();
    };

    const seoConfig = {
        title: t('book'),
    };

    const breadCrumbItems = [
        {
            label: t('layout:book'),
            path: '/',
        },
    ];

    return (
        <Layout staff={staff} breadCrumbItems={breadCrumbItems} activeMenu={['book']} seoConfig={seoConfig} withBackground>
            <div className="flex justify-between mb-4 flex-col sm:flex-row">
                <div className="flex flex-row ">
                    <FilterDrawer
                        filterBookForm={filterBookForm}
                        onSearch={() => bookListQuery.refetch()}
                        onReset={onResetFilterHandler}
                        loading={bookListQuery.isLoading}
                    />
                    <Button type="link" className="list-none" onClick={onResetFilterHandler}>
                        {t('common:reset-filter')}
                    </Button>
                </div>
                <div className="mt-2 sm:mt-0">
                    {staff.role.BOOK_CREATE && (
                        <Button icon={<PlusCircleOutlined />} type="primary" onClick={() => setIsAddBookModalOpen(true)}>
                            {t('add-book')}
                        </Button>
                    )}
                </div>
            </div>
            <Table
                columns={columns}
                dataSource={bookListQuery.data}
                loading={bookListQuery.isLoading}
                rowKey={(record) => record.id}
                scroll={{ x: 1000 }}
                onChange={paginationOnChange}
                pagination={{
                    current: pagination.page,
                    pageSize: pagination.pageSize,
                    defaultPageSize: 1,
                    showSizeChanger: true,
                    pageSizeOptions: [10, 25, 50, 100],
                    showTotal: (total, range) => t('common:pagination', { range0: range[0], range1: range[1], total: total }),
                    total: pagination.total,
                }}
            />
            <AddBook setOpen={setIsAddBookModalOpen} open={isAddBookModalOpen} refetch={() => bookListQuery.refetch()} />
        </Layout>
    );
};

export const getServerSideProps: GetServerSideProps = async ({ req, locale, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req, 'BOOK_VIEW');

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['book', 'layout', 'common', 'messages', 'book-request', 'api-messages'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};

export default Index;
