import { Button, Form, Modal } from 'antd';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { useTranslation } from 'next-i18next';
import { AxiosErrorResponse, EditStudyGroup } from '@/types';
import { toast } from 'react-toastify';
import { Dispatch, SetStateAction, useEffect } from 'react';
import { errorMessageFormatter } from '@/utils';
import EditStudyGroupForm from '../EditGroupForm';
import dayjs from 'dayjs';
import { getStudyGroupById, updateStudyGroupById } from '@/services/study-group';

interface EditStudyGroupModalProps {
    open: boolean;
    setOpen: Dispatch<SetStateAction<boolean>>;
    refetch: () => void;
    studyGroupId: string | undefined;
}

const EditStudyGroupModal: React.FC<EditStudyGroupModalProps> = ({ open, setOpen, refetch, studyGroupId }) => {
    const { t } = useTranslation(['study-group', 'layout', 'common', 'messages']);
    const [updateStudyGroupForm] = Form.useForm();
    const queryClient = useQueryClient();

    const getStudyGroupQuery = useQuery({
        queryKey: ['study-group', 'study-group-id', studyGroupId],
        queryFn: async () => {
            const response = await getStudyGroupById(studyGroupId as string);

            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    const studyGroupData = getStudyGroupQuery.data;

    useEffect(() => {
        if (studyGroupData) {
            updateStudyGroupForm.setFieldsValue({
                name: studyGroupData.name,
                groupPeriod: [dayjs.tz(studyGroupData.startDate), dayjs.tz(studyGroupData.endDate)],
                description: studyGroupData?.description,
            });
        }
    });

    const updateStudyGroupMutation = useMutation({
        mutationFn: async (body: EditStudyGroup) => {
            const response = await updateStudyGroupById(studyGroupId as string, body);
            return response;
        },
        onSuccess: () => {
            onCancelHandler();
            refetch();
            queryClient.invalidateQueries(['activityLog']);
        },
    });

    const onCancelHandler = () => {
        updateStudyGroupForm.resetFields();
        setOpen(false);
    };

    const onEditStudyGroupHandler = async () => {
        updateStudyGroupForm.validateFields().then(async (values) => {
            const { name, groupPeriod, description } = values;
            toast.promise(updateStudyGroupMutation.mutateAsync({ name, groupPeriod, description }), {
                pending: t('messages:updating-group'),
                success: t('messages:group-updated'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('edit-group')}>
            <Form form={updateStudyGroupForm} layout="vertical">
                <EditStudyGroupForm studyGroupId={studyGroupId} studyGroupData={studyGroupData} />
                <div className="flex justify-between gap-4">
                    <Button loading={updateStudyGroupMutation.isLoading} block size="large" onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button loading={updateStudyGroupMutation.isLoading} block size="large" type="primary" onClick={onEditStudyGroupHandler}>
                        {t('update-group')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default EditStudyGroupModal;
