import { Button, Form, Modal } from 'antd';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { useTranslation } from 'next-i18next';
import { AxiosErrorResponse, MemberStatus, SortOrder } from '@/types';
import { toast } from 'react-toastify';
import { Dispatch, SetStateAction } from 'react';
import { errorMessageFormatter } from '@/utils';
import { addMember, getMemberListByPagination } from '@/services/study-group';
import { useRouter } from 'next/router';
import DebounceSelect from '@/components/DebounceSelect';

interface AddMemberModalProps {
    open: boolean;
    setOpen: Dispatch<SetStateAction<boolean>>;
    refetch: () => void;
}

const AddMember: React.FC<AddMemberModalProps> = ({ open, setOpen, refetch }) => {
    const { t } = useTranslation(['study-group', 'layout', 'common', 'messages']);
    const [addMemberForm] = Form.useForm();
    const router = useRouter();
    const { studyGroupId } = router.query;
    const queryClient = useQueryClient();

    // Query
    const fetchDebounceMember = async (search: string) => {
        try {
            const response = await getMemberListByPagination(
                {
                    page: 1,
                    pageSize: 10,
                    sortField: 'fullName',
                    sortOrder: SortOrder.ASC,
                    fullName: search,
                    status: [MemberStatus.ACTIVE],
                },
                studyGroupId as string,
            );
            const { rows } = response.data;
            const options = rows.map((row) => ({
                label: row.preferredName !== null ? row.fullName + ' (' + row.preferredName + ')' : row.fullName,
                value: row.id,
            }));
            return options;
        } catch (error: any) {
            toast.error(t(errorMessageFormatter(error.message)));
        }
        return [];
    };

    // Mutation
    const addMemberMutation = useMutation({
        mutationFn: async (memberId: string) => {
            const response = await addMember(studyGroupId as string, memberId);
            return response;
        },
        onSuccess: () => {
            queryClient.invalidateQueries(['study-group', studyGroupId, 'study-group-member']);
            onCancelHandler();
            refetch();
        },
    });

    // Function
    const onCancelHandler = () => {
        addMemberForm.resetFields();
        setOpen(false);
    };

    const onAddMemberHandler = async () => {
        addMemberForm.validateFields().then(async (values) => {
            toast.promise(addMemberMutation.mutateAsync(values?.name?.value), {
                pending: t('messages:adding-member'),
                success: t('messages:member-added'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('add-member')}>
            <Form form={addMemberForm} layout="vertical">
                <Form.Item
                    rules={[
                        {
                            required: true,
                        },
                    ]}
                    name={'name'}
                    label={t('name')}
                >
                    <DebounceSelect fetchOptions={fetchDebounceMember} size="large" />
                </Form.Item>
                <div className="flex justify-between gap-4">
                    <Button loading={addMemberMutation.isLoading} block size="large" onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button
                        loading={addMemberMutation.isLoading}
                        block
                        size="large"
                        type="primary"
                        onClick={() => {
                            onAddMemberHandler();
                        }}
                    >
                        {t('add-member')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default AddMember;
