import { useStudyGroupContext } from '@/providers/StudyGroupContext';
import { Alert, Button, Empty, Form, Tabs, Tag } from 'antd';
import { useTranslation } from 'next-i18next';
import { useQuery, useQueryClient } from '@tanstack/react-query';
import { toast } from 'react-toastify';
import { AxiosErrorResponse, Staff, StudyGroupStatus } from '@/types';
import { conditionalReturn, currencyFormatter, errorMessageFormatter, mediaUrlGenerator } from '@/utils';
import PenaltyTab from './tabs/Penalty';
import { useEffect, useState } from 'react';
import EditGroup from './modals/EditGroup';
import Layout from '../layouts';
import { getStudyGroupPenalty } from '@/services/study-group';
import Task from './tabs/task';
import Member from './tabs/member';
import HtmlParser from '../HtmlParser';
import ActivityLog from '../activityLog';
import Link from 'next/link';
import { dateTransformer } from '@/utils/timeTransformer';

interface StudyGroupModuleProps {
    staff: Staff;
}

const StudyGroupModule: React.FC<StudyGroupModuleProps> = ({ staff }) => {
    const { t } = useTranslation(['study-group', 'common', 'messages', 'layout']);
    const { studyGroupId, studyGroupMember, studyGroup } = useStudyGroupContext();
    const [isEditGroupModalOpen, setIsEditGroupModalOpen] = useState<boolean>(false);
    const queryClient = useQueryClient();

    const studyGroupData = studyGroup.data;
    const studyGroupMemberData = studyGroupMember.data;

    const bookLink = studyGroupData?.book?.link;

    useEffect(() => {
        studyGroupDescriptionForm.setFieldsValue({
            description: studyGroupData?.description,
        });
    });

    // Form Instance
    const [studyGroupDescriptionForm] = Form.useForm();
    const [studyGroupDataForm] = Form.useForm();

    // Query
    const { data: studyGroupPenaltyData } = useQuery({
        queryKey: ['task-penalty', studyGroupId],
        queryFn: async () => {
            const response = await getStudyGroupPenalty(studyGroupId);
            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Function
    const renderGroupStatus = (status: StudyGroupStatus | undefined) => {
        switch (status) {
            case StudyGroupStatus.DRAFT:
                return (
                    <Tag color="#CCCCCC" className="w-fit">
                        {t('draft')}
                    </Tag>
                );
            case StudyGroupStatus.GENERATED:
                return (
                    <Tag color="#0074D9" className="w-fit">
                        {t('generated')}
                    </Tag>
                );
            case StudyGroupStatus.ONGOING:
                return (
                    <Tag color="#FFD700" className="w-fit">
                        {t('ongoing')}
                    </Tag>
                );
            case StudyGroupStatus.COMPLETED:
                return (
                    <Tag color="#2ECC40" className="w-fit">
                        {t('completed')}
                    </Tag>
                );
            default:
                return null;
        }
    };

    const onTabChangeHandler = (key: string) => {
        if (key === 'activityLog') {
            queryClient.invalidateQueries(['activityLog']);
        }
    };

    // Data
    const tabItems = [
        // Task Tab
        {
            label: t('task'),
            key: 'task',
            children: <Task />,
        },
        // Member Tab
        {
            label: t('member'),
            key: 'member',
            children: <Member />,
        },
        // Penalty Tab
        {
            label: t('penalty'),
            key: 'penalty',
            children: <PenaltyTab />,
        },
        ...conditionalReturn(staff.role.ACTIVITY_LOG, [
            {
                key: 'activityLog',
                label: t('common:activity-log'),
                children: <ActivityLog target={`study-group:${studyGroupId}`} />,
            },
        ]),
    ];

    const breadCrumbItems = [
        {
            label: t('study-group'),
            path: '/study-group',
        },
        {
            label: studyGroupData?.name ?? t('common:no-data'),
            path: '/',
        },
    ];

    const seoConfig = {
        title: t('study-group'),
    };

    return (
        <Layout breadCrumbItems={breadCrumbItems} activeMenu={['study-group']} seoConfig={seoConfig} staff={staff} activeDropdown={['study-group']}>
            <div className="bg-white p-9 pb-4 mb-4 rounded-lg">
                {!studyGroupData?.description && (
                    <div className="mb-2">
                        <Alert
                            message={t('common:add-description')}
                            description={t('common:add-a-description-by-updating-details')}
                            type="info"
                            showIcon
                        />
                    </div>
                )}
                <Form form={studyGroupDataForm}>
                    <p className="text-lg md:text-2xl text-[#544d4d] mb-2">
                        <b>{studyGroupData?.name}</b>
                    </p>
                    <p className="text-xs md:text-sm text-[#919191] w-full md:w-fit">
                        <div className="grid grid-cols-1 sm:grid-cols-4 gap-2">
                            <span className="font-bold col-span-1">{t('total-members')}: </span>
                            <p className="col-span-3">{studyGroupMemberData?.memberCount}</p>
                            <span className="font-bold col-span-1">{t('period')}:</span>
                            <p className="col-span-3">
                                {dateTransformer(studyGroupData?.startDate ?? '')} - {dateTransformer(studyGroupData?.endDate ?? '')}
                            </p>
                            <span className="font-bold col-span-1">{t('penalties')}:</span>
                            <p className="col-span-3">MYR {currencyFormatter(studyGroupPenaltyData)}</p>
                            <span className="font-bold col-span-1">{t('common:status')}:</span>
                            <p className="col-span-3">{renderGroupStatus(studyGroupData?.status)}</p>
                            <span className="font-bold col-span-1">{t('book-name')}:</span>
                            <p className="col-span-3">
                                <Link href={`/book/${studyGroupData?.book.id}`}>{studyGroupData?.book.name}</Link>
                            </p>
                            {studyGroupData?.book.uploadType === 'LINK' && <span className="font-bold col-span-1">{t('book-link')}:</span>}
                            {studyGroupData?.book.uploadType === 'FILE' && <span className="font-bold col-span-1">{t('book-file')}:</span>}

                            {studyGroupData?.book.uploadType === 'LINK' ? (
                                <div className="col-span-3">
                                    <a target="_blank" href={`${bookLink}`}>
                                        {bookLink}
                                    </a>
                                </div>
                            ) : (
                                studyGroupData?.book.uploadType === 'FILE' && (
                                    <div className="flex flex-col col-span-3">
                                        {studyGroupData?.book?.bookMedias.map((media) => {
                                            return (
                                                <>
                                                    <a target="_blank" href={mediaUrlGenerator(media.media.key as string)} key={media.media.id}>
                                                        {media.media.name}
                                                    </a>
                                                </>
                                            );
                                        })}
                                    </div>
                                )
                            )}
                            {studyGroupData?.description && (
                                <>
                                    <div className="font-bold col-span-1 text-[#919191]">{t('description')}:</div>
                                    <div className="col-span-3 text-[#919191] border-solid rounded-lg border-[#e8e8e8]">
                                        <div className="p-2 text-[#919191] text-xs md:text-sm">
                                            <HtmlParser html={studyGroupData?.description ?? ''} />
                                        </div>
                                    </div>
                                </>
                            )}
                        </div>
                    </p>
                    {studyGroupData?.status !== 'COMPLETED' && (
                        <div className="w-full flex justify-end mt-2">
                            <Button
                                type="primary"
                                onClick={() => {
                                    setIsEditGroupModalOpen(true);
                                }}
                            >
                                {t('common:update-details')}
                            </Button>
                        </div>
                    )}
                </Form>
            </div>
            {!studyGroupData ? (
                <Empty />
            ) : (
                <div>
                    <Tabs items={tabItems} onChange={onTabChangeHandler} className="bg-white p-9 pt-5 rounded-lg" />
                </div>
            )}
            {isEditGroupModalOpen && (
                <EditGroup
                    open={isEditGroupModalOpen}
                    setOpen={setIsEditGroupModalOpen}
                    refetch={() => studyGroup.refetch()}
                    studyGroupId={studyGroupId}
                />
            )}
        </Layout>
    );
};

export default StudyGroupModule;
