import { Trans, useTranslation } from 'next-i18next';
import ConfirmationModal from '../modals/ConfirmationModal';
import { AxiosErrorResponse, StudyGroup } from '@/types';
import { toast } from 'react-toastify';
import { UseQueryResult, useMutation } from '@tanstack/react-query';
import { errorMessageFormatter } from '@/utils';
import { Button } from 'antd';
import { deleteGroupById } from '@/services/study-group';

interface DeleteGroupActionProps {
    studyGroup: StudyGroup;
    isTable?: boolean;
    query: UseQueryResult<StudyGroup[] | null>;
}

const DeleteGroupAction: React.FC<DeleteGroupActionProps> = ({ studyGroup, isTable = false, query }) => {
    const { t } = useTranslation(['group', 'layout', 'common', 'messages']);

    const deleteGroupMutation = useMutation({
        mutationFn: async () => {
            const response = await deleteGroupById(studyGroup.id);
            return response.data;
        },
        onSuccess: () => {
            query.refetch();
        },
    });

    const onDeleteHandler = () => {
        toast.promise(deleteGroupMutation.mutateAsync(), {
            pending: t('messages:deleting-group'),
            success: t('messages:group-deleted'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    return (
        <ConfirmationModal
            title={t('common:delete-confirmation')}
            message={
                <Trans i18nKey={'messages:are-you-sure-you-want-to-delete'} components={{ strong: <strong /> }} values={{ name: studyGroup.name }} />
            }
            okText={t('common:delete')}
            onOk={onDeleteHandler}
            reason={false}
            okButtonProps={{
                danger: true,
            }}
            width={400}
        >
            {isTable ? (
                <Button
                    loading={deleteGroupMutation.isLoading}
                    type="text"
                    block
                    className="!p-0 hover:!text-white hover:bg-none"
                    style={{ transition: 'none' }}
                >
                    {t('common:delete')}
                </Button>
            ) : (
                <Button
                    loading={deleteGroupMutation.isLoading}
                    type="link"
                    ghost
                    block
                    className="!p-0 hover:!text-white text-black"
                    style={{ transition: 'none' }}
                >
                    {t('common:delete')}
                </Button>
            )}
        </ConfirmationModal>
    );
};

export default DeleteGroupAction;
