import { Button, Descriptions, Drawer, Form, Tabs } from 'antd';
import { useTranslation } from 'next-i18next';
import CommentTab from '../tabs/Comment';
import PenaltyTab from '../tabs/Penalty';
import React, { useEffect } from 'react';
import { EditOutlined } from '@ant-design/icons';
import { useStudyGroupContext } from '@/providers/StudyGroupContext';
import { toast } from 'react-toastify';
import { useMutation, useQuery } from '@tanstack/react-query';
import { currencyFormatter, errorMessageFormatter } from '@/utils';
import { AxiosErrorResponse } from '@/types';
import { getTaskPenalty, updateDescription } from '@/services/study-group';
import { Editor } from '@tinymce/tinymce-react';
import HtmlParser from '@/components/HtmlParser';
import { dateTransformer } from '@/utils/timeTransformer';
import { v4 as uuid } from 'uuid';

const DItem = Descriptions.Item;

interface CommentsProps {
    open: boolean;
    setOpen: (open: boolean) => void;
}

const PreviewTask: React.FC<CommentsProps> = ({ open, setOpen }) => {
    const { t, i18n } = useTranslation(['study-group', 'layout', 'common', 'messages']);
    const currentLocale = i18n.language;
    const [editDescription, setEditDescription] = React.useState<boolean>(false);

    const { studyGroupTask, studyGroupTaskId } = useStudyGroupContext();

    const studyGroupTaskData = studyGroupTask.data;

    // Form Instance
    const [updateStudyGroupTaskDescriptionForm] = Form.useForm();

    // Reset Edit Description if modal is closed
    useEffect(() => {
        if (open) {
            setEditDescription(false);
        }
    }, [open]);

    useEffect(() => {
        updateStudyGroupTaskDescriptionForm.setFieldsValue({
            description: studyGroupTaskData?.description,
        });
    }, [studyGroupTaskData]);

    // Query
    const { data: taskPenaltyData } = useQuery({
        queryKey: ['task-penalty', studyGroupTaskId],
        enabled: !!studyGroupTaskId,
        queryFn: async () => {
            const response = await getTaskPenalty(studyGroupTaskId);
            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Mutation
    const editDescriptionMutation = useMutation({
        mutationFn: async () => {
            const values = updateStudyGroupTaskDescriptionForm.getFieldsValue().description;
            const res = await updateDescription(studyGroupTaskId, values);

            return res.data;
        },
    });

    // Function
    const onEditDescriptionHandler = () => {
        toast.promise(
            (async () => {
                await editDescriptionMutation.mutateAsync();
                await studyGroupTask.refetch();
                setEditDescription(false);
            })(),
            {
                pending: t('messages:updating-description'),
                success: t('messages:description-updated'),
                error: {
                    render: ({ data }) => t(errorMessageFormatter(data as AxiosErrorResponse)),
                },
            },
        );
    };

    // Data
    const tabItems = [
        // Comment Tab
        {
            label: t('comment'),
            key: 'comment',
            children: <CommentTab />,
        },
        // Penalty Tab
        {
            label: t('penalty'),
            key: 'penalty',
            children: <PenaltyTab />,
        },
    ];

    return (
        <Drawer
            open={open}
            onClose={() => {
                setOpen(false);
            }}
            placement="bottom"
            height="90%"
        >
            <Descriptions bordered size="small" className="mb-4" column={1} labelStyle={{ maxWidth: '120px' }}>
                <DItem label={t('description')} contentStyle={{ fontWeight: 'normal' }}>
                    <div className="text-justify">
                        <div className="w-full">
                            {editDescription ? (
                                <div className="flex flex-col w-full">
                                    <Form form={updateStudyGroupTaskDescriptionForm} layout="vertical">
                                        <Form.Item name="description" trigger="onEditorChange" noStyle>
                                            <Editor
                                                id={uuid()}
                                                tinymceScriptSrc={'/tinymce/js/tinymce/tinymce.min.js'}
                                                init={{
                                                    height: 200,
                                                    menubar: '',
                                                    plugins: [
                                                        'advlist',
                                                        'autolink',
                                                        'lists',
                                                        'link',
                                                        'searchreplace',
                                                        'code',
                                                        'preview',
                                                        'wordcount',
                                                        'table',
                                                        'powerpaste',
                                                    ],
                                                    toolbar:
                                                        'undo redo | blocks | ' +
                                                        'paste' +
                                                        'bold italic forecolor | link | alignleft aligncenter ' +
                                                        'alignright alignjustify | bullist numlist outdent indent | ' +
                                                        'fullscreen',
                                                    block_formats: 'Paragraph=p;Header 2=h2;Header 3=h3;Header 4=h4',
                                                    content_style: 'body { font-family:var(--font-ubuntu); font-size:14px; text-align:left }',
                                                    language: currentLocale === 'en-GB' ? undefined : currentLocale,
                                                    language_url:
                                                        currentLocale === 'en-GB' ? undefined : '/editor-translation/' + currentLocale + '.js',
                                                    promotion: false,
                                                    paste_data_images: true,
                                                    keep_styles: true,
                                                    contextmenu: false,
                                                }}
                                            />
                                        </Form.Item>
                                        <div className="w-full flex justify-end gap-2 mt-2">
                                            <Button
                                                loading={editDescriptionMutation.isLoading}
                                                danger
                                                onClick={() => {
                                                    setEditDescription(false);
                                                }}
                                            >
                                                {t('cancel')}
                                            </Button>
                                            <Button loading={editDescriptionMutation.isLoading} type="primary" onClick={onEditDescriptionHandler}>
                                                {t('update')}
                                            </Button>
                                        </div>
                                    </Form>
                                </div>
                            ) : (
                                <div className="flex justify-between">
                                    <HtmlParser html={studyGroupTaskData?.description ?? ''} />
                                    {!studyGroupTaskData?.status && (
                                        <EditOutlined
                                            className="ml-auto pl-4 pointer"
                                            onClick={() => {
                                                setEditDescription(true);
                                            }}
                                        />
                                    )}
                                </div>
                            )}
                        </div>
                    </div>
                </DItem>
                <DItem label={t('task-date')} contentStyle={{ fontWeight: 'normal' }}>
                    <div className="flex justify-between">{dateTransformer(studyGroupTaskData?.taskDate ?? '')}</div>
                </DItem>
                <DItem label={t('penalty')} contentStyle={{ fontWeight: 'normal' }}>
                    MYR {currencyFormatter(taskPenaltyData?.totalAmount)}
                </DItem>
            </Descriptions>

            <Tabs items={tabItems} className="bg-white pt-2 rounded-lg" />
        </Drawer>
    );
};

export default PreviewTask;
