import { useEffect } from 'react';
import PhoneInput from 'react-phone-input-2';
import 'react-phone-input-2/lib/style.css';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { UseQueryResult, useMutation } from '@tanstack/react-query';
import { Button, Form, Modal } from 'antd';
import { updateStaffPhoneNumber } from '@/services/staff';
import { BaseModalProps, Staff, UpdateStaffPhoneNumberParams } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';

interface PhoneNumberModalProps extends BaseModalProps {
    staff: Staff;
    staffQuery: UseQueryResult<Staff | null>;
}

const PhoneNumberModal: React.FC<PhoneNumberModalProps> = ({ open, setOpen, staff, staffQuery }) => {
    const { t } = useTranslation(['staff', 'common']);
    const [form] = Form.useForm<{ phoneNumber: string }>();
    const phoneNumberInput = Form.useWatch('phoneNumber', form);

    useEffect(() => {
        if (open) {
            form.setFieldsValue({ phoneNumber: staff.phoneNumber });
        }
    }, [open]);

    const isDiasbled = () => {
        if (!phoneNumberInput || phoneNumberInput === staff.phoneNumber) {
            return true;
        }

        return false;
    };

    // Mutations
    const updateStaffPhoneNumberMutation = useMutation({
        mutationFn: async ({ staffId, body }: { staffId: string; body: UpdateStaffPhoneNumberParams }) => {
            const res = await updateStaffPhoneNumber(staffId, body);
            return res.data;
        },
        onSuccess: () => {
            setOpen(false);
            staffQuery.refetch();
        },
    });

    // Functions
    const onCancelHandler = () => {
        form.resetFields();
        setOpen(false);
    };

    const onUpdateStaffPhoneNumberHandler = () => {
        form.validateFields().then(async (values) => {
            toast.promise(updateStaffPhoneNumberMutation.mutateAsync({ staffId: staff.id, body: values }), {
                pending: t('messages:updating-staff-phone-number'),
                success: t('messages:staff-phone-number-updated'),
                error: {
                    render({ data }: any) {
                        return t(errorMessageFormatter(data));
                    },
                },
            });
        });
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('edit-phone-number')}>
            <Form layout="vertical" form={form} className="mt-4">
                <Form.Item label={t('phone-number')} name="phoneNumber" rules={[{ required: true }]} labelCol={{ flex: '35px' }}>
                    <PhoneInput
                        value={phoneNumberInput}
                        inputProps={{
                            required: true,
                        }}
                        inputClass="form-control"
                        country={'my'}
                        inputStyle={{ width: '100%', height: '40px', borderRadius: '8px' }}
                        buttonStyle={{ borderRadius: '8px 0px 0px 8px' }}
                    />
                </Form.Item>
                <div className="pt-2 flex gap-3">
                    <Button loading={updateStaffPhoneNumberMutation.isLoading} size="large" block onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button
                        loading={updateStaffPhoneNumberMutation.isLoading}
                        size="large"
                        type="primary"
                        disabled={isDiasbled()}
                        block
                        onClick={onUpdateStaffPhoneNumberHandler}
                    >
                        {t('common:update')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default PhoneNumberModal;
