import { useEffect } from 'react';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { UseQueryResult, useMutation } from '@tanstack/react-query';
import { Button, Form, Input, Modal } from 'antd';
import { updateStaffEmail } from '@/services/staff';
import { BaseModalProps, Staff, UpdateStaffEmailParams } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';

interface EmailModalProps extends BaseModalProps {
    staff: Staff;
    staffQuery: UseQueryResult<Staff | null>;
}

const EmailModal: React.FC<EmailModalProps> = ({ open, setOpen, staff, staffQuery }) => {
    const { t } = useTranslation(['staff', 'common']);
    const [form] = Form.useForm<{ email: string }>();
    const emailInput = Form.useWatch('email', form);

    useEffect(() => {
        if (open) {
            form.setFieldsValue({ email: staff.email });
        }
    }, [open]);

    const isDiasbled = () => {
        if (!emailInput || emailInput === staff.email) {
            return true;
        }

        return false;
    };

    // Mutations
    const updateStaffEmailMutation = useMutation({
        mutationFn: async ({ staffId, body }: { staffId: string; body: UpdateStaffEmailParams }) => {
            const res = await updateStaffEmail(staffId, body);
            return res.data;
        },
        onSuccess: () => {
            staffQuery.refetch();
        },
    });

    // Functions
    const onCancelHandler = () => {
        form.resetFields();
        setOpen(false);
    };

    const onUpdateStaffEmailHandler = () => {
        form.validateFields().then(async (values) => {
            toast.promise(updateStaffEmailMutation.mutateAsync({ staffId: staff.id, body: values }), {
                pending: t('messages:updating-staff-email'),
                success: {
                    render() {
                        setOpen(false);
                        staffQuery.refetch();
                        return t('messages:staff-email-updated');
                    },
                },
                error: {
                    render({ data }: any) {
                        return t(errorMessageFormatter(data));
                    },
                },
            });
        });
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('edit-email')}>
            <Form layout="vertical" form={form} className="mt-4">
                <Form.Item
                    label="New Email"
                    name="email"
                    rules={[
                        { required: true },
                        {
                            type: 'email',
                        },
                    ]}
                >
                    <Input size="large" />
                </Form.Item>
                <div className="pt-2 flex gap-3">
                    <Button size="large" block onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button size="large" type="primary" disabled={isDiasbled()} block onClick={onUpdateStaffEmailHandler}>
                        {t('common:update')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default EmailModal;
