import { Dispatch, SetStateAction } from 'react';
import { toast } from 'react-toastify';
import Image from 'next/image';
import Link from 'next/link';
import { useRouter } from 'next/router';
import { useMutation } from '@tanstack/react-query';
import { Avatar, Button, Dropdown } from 'antd';
import { logout } from '@/services/auth';
import { AxiosErrorResponse } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';
import { stringAvatar, stringToColor } from '@/utils/helperFunctions';
import Logo from '../../../public/logo-text.png';
import LanguageSwitcher from '../globals/LanguageSwitcher';
import { useTranslation } from 'next-i18next';

export const Header = ({
    isMobile,
    isDrawerOpen,
    setIsDrawerOpen,
    collapsed,
    setCollapsed,
    fullName,
}: {
    isMobile: boolean;
    isDrawerOpen: boolean;
    setIsDrawerOpen: Dispatch<SetStateAction<boolean>>;
    collapsed: boolean;
    setCollapsed: Dispatch<SetStateAction<boolean>>;
    fullName: string;
}) => {
    const { t } = useTranslation(['my-profile', 'layout', 'common', 'messages']);
    const router = useRouter();

    const logoutMutation = useMutation({
        mutationFn: async () => {
            await logout();
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(errorMessageFormatter(error));
        },
        onSuccess: () => {
            router.push('/');
        },
    });

    return (
        <div
            style={{
                borderBottom: '1px solid #e8e8e8',
                position: 'fixed',
                width: '100%',
                zIndex: 100,
            }}
            className="h-16 px-6 bg-white flex justify-between items-center"
        >
            <div className="flex items-center gap-4">
                <Button
                    onClick={() => {
                        if (isMobile) {
                            setIsDrawerOpen(!isDrawerOpen);
                        } else {
                            setCollapsed(!collapsed);
                        }
                    }}
                    aria-controls="sidebar"
                    className="w-8 flex justify-center items-center py-0 px-1"
                >
                    <svg className="w-5 h-5 text-gray-500" fill="currentColor" viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg">
                        <path
                            fillRule="evenodd"
                            d="M3 5a1 1 0 011-1h12a1 1 0 110 2H4a1 1 0 01-1-1zM3 10a1 1 0 011-1h6a1 1 0 110 2H4a1 1 0 01-1-1zM3 15a1 1 0 011-1h12a1 1 0 110 2H4a1 1 0 01-1-1z"
                            clipRule="evenodd"
                        ></path>
                    </svg>
                </Button>
                {!isMobile && (
                    <div className="w-[130px] mt-2">
                        <Image src={Logo} alt="Image Logo" className="w-full h-full" />
                    </div>
                )}
            </div>

            <div className="flex items-center justify-center">
                <div className="flex items-center justify-center col-row">
                    <Dropdown
                        trigger={['click']}
                        dropdownRender={() => {
                            return (
                                <div className="flex flex-col p-3 bg-white rounded-md shadow-md">
                                    <LanguageSwitcher />
                                    {/* <Divider className="my-0" /> */}
                                    <Link href="/my-profile" className="leading-[normal]">
                                        <div className="py-2 pl-3 mt-1 duration-100 hover:bg-slate-100 rounded-[4px] cursor-pointer">
                                            {t('layout:my-profile')}
                                        </div>
                                    </Link>
                                    <div
                                        className="w-full text-center bg-red-600 rounded-[4px] hover:bg-red-700 duration-100 py-2 mt-2 cursor-pointer leading-[normal]"
                                        onClick={() => {
                                            toast.success(t('messages:logout-successfully'));
                                            logoutMutation.mutate();
                                        }}
                                    >
                                        <a className="text-white hover:text-white">{t('layout:logout')}</a>
                                    </div>
                                </div>
                            );
                        }}
                    >
                        <div className="cursor-pointer flex items-center">
                            <Avatar style={{ backgroundColor: stringToColor(fullName) }} size={35} className="flex items-center">
                                {stringAvatar(fullName)}
                            </Avatar>
                            <div className="ml-2 flex items-center gap-2">
                                <span className="font-semibold">{fullName}</span>
                                <svg
                                    xmlns="http://www.w3.org/2000/svg"
                                    fill="none"
                                    viewBox="0 0 24 24"
                                    strokeWidth="1.5"
                                    stroke="currentColor"
                                    className="w-4 h-4"
                                >
                                    <path strokeLinecap="round" strokeLinejoin="round" d="M19.5 8.25l-7.5 7.5-7.5-7.5" />
                                </svg>
                            </div>
                        </div>
                    </Dropdown>
                </div>
            </div>
        </div>
    );
};
