import React, { useState } from 'react';
import { useTranslation } from 'next-i18next';
import { FilterOutlined, SearchOutlined } from '@ant-design/icons';
import { Button, Col, DatePicker, Form, Input, Modal, Row, Select } from 'antd';
import { FormInstance } from 'antd/es/form/Form';
import PhoneInput from 'react-phone-input-2';
import 'react-phone-input-2/lib/style.css';

interface FilterAttributes {
    filterStudyRequestForm: FormInstance;
    onReset: () => void;
    onSearch: () => void;
    loading: boolean;
    filterByBookName?: boolean;
}

const { RangePicker } = DatePicker;

const FilterDrawer: React.FC<FilterAttributes> = ({ filterStudyRequestForm, onReset, onSearch, loading, filterByBookName }) => {
    const { t } = useTranslation(['book', 'common']);
    const criteriaSelected = filterStudyRequestForm.getFieldsValue();
    const criteriaCount: number = Object.values(criteriaSelected).reduce((count: number, val) => count + (val ? 1 : 0), 0);
    const [isModalOpen, setIsModalOpen] = useState(false);

    const breakPoint = {
        xs: 24,
        sm: 24,
        md: 12,
        lg: 12,
    };

    const onResetHandler = () => {
        onReset();
        setIsModalOpen(false);
    };

    const onSearchHandler = () => {
        onSearch();
        setIsModalOpen(false);
    };

    return (
        <div>
            <Button onClick={() => setIsModalOpen(true)} icon={<FilterOutlined />}>
                {criteriaCount > 0 && `(${criteriaCount})`} {t('common:filter')}
            </Button>
            <Modal title={t('common:filter')} onCancel={onResetHandler} open={isModalOpen} footer={null} width={650}>
                <Form form={filterStudyRequestForm} layout="vertical" name="filter_form" className="mt-6" size="large">
                    <Row gutter={[16, 0]}>
                        {/* Name */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="name" label={t('member-name')} labelCol={{ flex: '35px' }}>
                                <Input placeholder={t('enter-member-name')} />
                            </Form.Item>
                        </Col>
                        {/* Book Name */}
                        {filterByBookName && (
                            <Col {...breakPoint}>
                                <Form.Item initialValue="" name="bookName" label={t('book-name')} labelCol={{ flex: '35px' }}>
                                    <Input placeholder={t('enter-book-name')} />
                                </Form.Item>
                            </Col>
                        )}
                        {/* Email */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="email" label={t('email')} labelCol={{ flex: '35px' }}>
                                <Input placeholder={t('enter-an-email')} />
                            </Form.Item>
                        </Col>
                        {/* Phone Number */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="phoneNumber" label={t('phone-number')} labelCol={{ flex: '35px' }}>
                                <PhoneInput
                                    placeholder={t('enter-a-phone-number')}
                                    inputProps={{
                                        required: true,
                                    }}
                                    inputClass="form-control"
                                    country={'my'}
                                    inputStyle={{ width: '100%', height: '40px', borderRadius: '8px' }}
                                    buttonStyle={{ borderRadius: '8px 0px 0px 8px' }}
                                />
                            </Form.Item>
                        </Col>
                        {/* Created At */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="createdAt" label={t('common:created-at')} labelCol={{ flex: '35px' }}>
                                <RangePicker className="!w-full" />
                            </Form.Item>
                        </Col>
                        {/* Status */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="status" label={t('status')} labelCol={{ flex: '35px' }}>
                                <Select placeholder={t('common:select')}>
                                    <Select.Option value="">{t('select-a-status')}</Select.Option>
                                    <Select.Option value="JOINED">{t('common:joined')}</Select.Option>
                                    <Select.Option value="PENDING">{t('common:pending')}</Select.Option>
                                    <Select.Option value="REJECTED">{t('common:rejected')}</Select.Option>
                                </Select>
                            </Form.Item>
                        </Col>
                    </Row>
                    <div className="flex justify-between mt-3 gap-x-3">
                        <Button onClick={onResetHandler} block loading={loading}>
                            {t('common:reset')}
                        </Button>
                        <Button type="primary" onClick={onSearchHandler} block loading={loading}>
                            {t('common:apply-filter')}
                            <SearchOutlined />
                        </Button>
                    </div>
                </Form>
            </Modal>
        </div>
    );
};

export default FilterDrawer;
