import { Button, Form, Modal } from 'antd';
import { useMutation } from '@tanstack/react-query';
import { addBook } from '@/services/book';
import BookFormFields from '@/components/book/BookFormFields';
import { useTranslation } from 'next-i18next';
import { AxiosErrorResponse, Book, UploadType } from '@/types';
import { toast } from 'react-toastify';
import { Dispatch, SetStateAction } from 'react';
import { errorMessageFormatter } from '@/utils';
import { UploadFile } from 'antd/lib';
import { mediaFormValueToResponse } from '@/transformer/media/media';
import { conditionalReturn } from '@/utils';

interface AddBookModalProps {
    open: boolean;
    setOpen: Dispatch<SetStateAction<boolean>>;
    refetch: () => void;
}

const AddBook: React.FC<AddBookModalProps> = ({ open, setOpen, refetch }) => {
    const { t } = useTranslation(['book', 'layout', 'common', 'messages']);
    const [addBookForm] = Form.useForm<Book & { uploadFile: UploadFile[] } & { bookImages: UploadFile[] }>();

    const addBookMutation = useMutation({
        mutationFn: async (book: Book & { uploadFile: UploadFile[]; bookImages: UploadFile[] }) => {
            const { uploadFile, bookImages, ...body } = book;
            const formattedData = book.uploadType === UploadType.FILE ? uploadFile.map(mediaFormValueToResponse) : null;
            const formattedImages = bookImages.map(mediaFormValueToResponse);

            const response = await addBook({
                ...body,
                ...conditionalReturn(book.uploadType === UploadType.FILE, { uploadFile: formattedData }),
                bookImages: formattedImages,
            });

            return response;
        },
        onSuccess: () => {
            onCancelHandler();
            refetch();
        },
    });

    const onCancelHandler = () => {
        addBookForm.resetFields();
        setOpen(false);
    };

    const onAddBookHandler = async () => {
        addBookForm.validateFields().then(async (values) => {
            toast.promise(addBookMutation.mutateAsync(values), {
                pending: t('messages:adding-book'),
                success: t('messages:book-added'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('add-book')}>
            <Form
                form={addBookForm}
                initialValues={{
                    price: 0,
                    uploadType: UploadType.NONE,
                }}
                layout="vertical"
            >
                <BookFormFields />
                <div className="flex justify-between gap-4">
                    <Button loading={addBookMutation.isLoading} block size="large" onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button loading={addBookMutation.isLoading} block size="large" type="primary" onClick={onAddBookHandler}>
                        {t('add-book')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default AddBook;
