import { Trans, useTranslation } from 'next-i18next';
import ConfirmationModal from '../modals/ConfirmationModal';
import { AxiosErrorResponse, Book } from '@/types';
import { toast } from 'react-toastify';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { bookKeys, deleteBookById } from '@/services/book';
import { useRouter } from 'next/router';
import { errorMessageFormatter } from '@/utils';
import { Button } from 'antd';

interface DeleteBookActionProps {
    book: Book;
    isTable?: boolean;
}

const DeleteBookAction: React.FC<DeleteBookActionProps> = ({ book, isTable = false }) => {
    const { t } = useTranslation(['book', 'layout', 'common', 'messages']);
    const router = useRouter();
    const queryClient = useQueryClient();

    const deleteBookMutation = useMutation({
        mutationFn: async () => {
            const response = await deleteBookById(book.id);
            return response.data;
        },
        onSuccess: () => {
            if (!isTable) {
                router.push('/book');
            } else {
                queryClient.invalidateQueries(bookKeys.pagination());
            }
        },
    });

    const onDeleteHandler = () => {
        toast.promise(deleteBookMutation.mutateAsync(), {
            pending: t('messages:deleting-book'),
            success: t('messages:book-deleted'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    return (
        <ConfirmationModal
            title={t('common:delete-confirmation')}
            message={<Trans i18nKey={'messages:are-you-sure-you-want-to-delete'} components={{ strong: <strong /> }} values={{ name: book.name }} />}
            okText={t('common:delete')}
            onOk={onDeleteHandler}
            reason={false}
            okButtonProps={{
                danger: true,
            }}
            width={400}
        >
            {isTable ? (
                <Button
                    loading={deleteBookMutation.isLoading}
                    type="text"
                    block
                    className="!p-0 hover:!text-white hover:!bg-transparent !w-fit !h-fit"
                    style={{ transition: 'none' }}
                >
                    {t('common:delete')}
                </Button>
            ) : (
                <Button
                    loading={deleteBookMutation.isLoading}
                    type="primary"
                    danger
                    block
                    className="!p-0 hover:!text-white"
                    style={{ transition: 'none' }}
                >
                    {t('common:delete')}
                </Button>
            )}
        </ConfirmationModal>
    );
};

export default DeleteBookAction;
