import { Col, Form, Input, Radio, Row, Select } from 'antd';
import { useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';
import { useQuery } from '@tanstack/react-query';
import { AxiosErrorResponse } from '@/types';
import { errorMessageFormatter } from '@/utils';
import { getStudyGroupListByBookId } from '@/services/book-request';
import { StudyGroup } from '@/types/study-group';
import PresetCalendarRange from '../PresetCalendarRange';
import { useState } from 'react';
import { Dayjs } from 'dayjs';

const { Option } = Select;

interface ApproveRequestFormFieldsProps {
    addGroup: boolean;
    setAddGroup: React.Dispatch<React.SetStateAction<boolean>>;
    bookId?: string;
}

const ApproveRequestFormFields: React.FC<ApproveRequestFormFieldsProps> = ({ addGroup, setAddGroup, bookId }) => {
    const { t } = useTranslation(['book-request', 'layout', 'common', 'messages']);
    const [selectedDateRange, setSelectedDateRange] = useState<[Dayjs, Dayjs] | undefined>();

    // Query
    const studyGroupListQuery = useQuery({
        queryKey: ['study-request', 'book', bookId],
        enabled: !addGroup,
        queryFn: async () => {
            const response = await getStudyGroupListByBookId(bookId as string);

            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    return (
        <Row gutter={[16, 16]} className="mb-4">
            <Col span={24}>
                <Radio.Group
                    className="w-full justify-between"
                    value={addGroup}
                    onChange={(e) => {
                        setAddGroup(e.target.value);
                    }}
                >
                    <Radio value={false}>{t('invite-to-existing-group')}</Radio>
                    <Radio value={true}>{t('add-new-group')}</Radio>
                </Radio.Group>
            </Col>
            {addGroup ? (
                <Col span={24}>
                    <Form.Item
                        name="groupName"
                        label={t('group-name')}
                        rules={addGroup ? [{ required: true, message: t('messages:required') }] : [{ required: false }]}
                    >
                        <Input placeholder={t('enter-group-name')} />
                    </Form.Item>
                    <Form.Item
                        name="groupPeriod"
                        label={t('group-period')}
                        rules={addGroup ? [{ required: true, message: t('messages:required') }] : [{ required: false }]}
                    >
                        <PresetCalendarRange
                            value={selectedDateRange}
                            onChange={(dates) => setSelectedDateRange(dates)}
                            onCalendarChange={(dates) => setSelectedDateRange(dates)}
                        />
                    </Form.Item>
                </Col>
            ) : (
                <Col span={24}>
                    <Form.Item
                        name="groupId"
                        label={t('group-name')}
                        rules={!addGroup ? [{ required: true, message: t('messages:required') }] : [{ required: false }]}
                    >
                        <Select placeholder={t('select-a-group')} className="w-full">
                            {studyGroupListQuery.data?.map((studyGroup: StudyGroup) => (
                                <Option key={studyGroup.id} value={studyGroup.id}>
                                    {studyGroup.name}
                                </Option>
                            ))}
                        </Select>
                    </Form.Item>
                </Col>
            )}
        </Row>
    );
};

export default ApproveRequestFormFields;
