import { CloudDownloadOutlined } from '@ant-design/icons';
import { Button } from 'antd';
import { useTranslation } from 'next-i18next';
import { AxiosErrorResponse, BookTokenRequestParams } from '@/types';
import { useQuery, useQueryClient } from '@tanstack/react-query';
import { getExportBookTokenSubscriptionList } from '@/services/book-token-request';
import { toast } from 'react-toastify';
import { currencyFormatter, errorMessageFormatter } from '@/utils';
import ExcelJS from 'exceljs';
import { dateTimeTransformer } from '@/utils/timeTransformer';

// PDF
interface ExportDataProps {
    query: BookTokenRequestParams;
}

const ExportData: React.FC<ExportDataProps> = ({ query }) => {
    const { t } = useTranslation(['book-order', 'layout', 'common', 'messages']);
    const queryClient = useQueryClient();

    useQuery({
        queryKey: ['book-token-request', 'export'],
        enabled: false,
        queryFn: async () => {
            const res = await getExportBookTokenSubscriptionList(query);

            return res.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    const workbook = new ExcelJS.Workbook();
    const worksheet = workbook.addWorksheet('Book Token Requests');

    // Columns
    worksheet.columns = [
        { header: 'Reference Number', key: 'no', width: 20 },
        { header: 'Package', key: 'package', width: 40 },
        { header: 'Description', key: 'description', width: 40 },
        { header: 'Member', key: 'member', width: 30 },
        { header: 'Phone Number', key: 'phoneNumber', width: 20 },
        { header: 'Address', key: 'address', width: 80 },
        { header: 'Quantity', key: 'quantity', width: 10 },
        { header: 'Price', key: 'price', width: 15 },
        { header: 'Payment Status', key: 'paymentStatus', width: 20 },
        { header: 'Payment Date', key: 'paymentDate', width: 20 },
        { header: 'Created At', key: 'createdAt', width: 20 },
    ];

    // Add Row Style
    worksheet.getRow(1).font = { bold: true };
    worksheet.getRow(1).alignment = { horizontal: 'center' };
    worksheet.getRow(1).fill = {
        type: 'pattern',
        pattern: 'solid',
        fgColor: { argb: 'FFA9A9A9' },
    };

    const generateExcelHandler = async (data: any) => {
        data?.map((request: any) => {
            const orderPackage = request.package;
            const orderMember = request.member;

            worksheet.addRow({
                no: request.displayId,
                package: orderPackage.name,
                description: orderPackage.description,
                member: orderMember.fullName,
                phoneNumber: `;${orderMember.phoneNumber}`,
                address: orderMember.address,
                quantity: request.quantity,
                price: `RM ${currencyFormatter(request.price)}`,
                paymentStatus: request.paymentStatus,
                paymentDate: request.paymentDate ? dateTimeTransformer(request.paymentDate) : '-',
                createdAt: request.createdAt,
            });
        });

        // Generate the Excel file
        await workbook.xlsx.writeBuffer().then((buffer: any) => {
            const blob = new Blob([buffer], { type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' });
            const link = document.createElement('a');
            const url = URL.createObjectURL(blob);
            link.href = url;
            link.setAttribute('download', `book-token-request.csv`);
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        });
    };

    const exportToCsvHandler = () => {
        queryClient.fetchQuery(['book-token-request', 'export']).then((res) => {
            generateExcelHandler(res);
        });
    };

    return (
        <Button type="primary" icon={<CloudDownloadOutlined />} onClick={exportToCsvHandler}>
            {t('export-csv')}
        </Button>
    );
};

export default ExportData;
