import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { Tag, Tooltip } from 'antd';
import dayjs from 'dayjs';
import { Member, MemberStatus, Staff, StaffStatus, TokenType } from '@/types';
import { CopyOutlined } from '@ant-design/icons';
import { MEMBER_URL, STAFF_URL } from '@/configs';

const UserStatus: React.FC<{ user: Staff | Member }> = ({ user }) => {
    const { t } = useTranslation(['common', 'messages']);

    async function copyToClipboard(text: string) {
        try {
            await navigator.clipboard.writeText(text);
            toast.success(t('messages:copied-to-clipboard'));
        } catch (e) {
            toast.error(t('messages:failed-to-copy'));
        }
    }

    if (user.password) {
        if (user.hasOwnProperty('roleId')) {
            const staff: Staff = user as Staff;
            return (
                <Tag color={staff?.status === StaffStatus.ACTIVE ? 'green' : 'red'}>{staff?.status === 'ACTIVE' ? t('active') : t('inactive')}</Tag>
            );
        } else {
            const member: Member = user as Member;
            const label = member?.status !== MemberStatus.ACTIVE ? t('inactive') : t('active');
            const color = member?.status !== MemberStatus.ACTIVE ? 'red' : 'success';
            return <Tag color={color}>{label}</Tag>;
        }
    }

    const verificationToken = user.tokens.filter((token) => token.type === TokenType.CONFIRMATION);

    if (!verificationToken || verificationToken.length === 0 || dayjs.tz(verificationToken[0].expiredAt).isBefore(dayjs.tz())) {
        return <Tag color="red">{t('verification-expired')}</Tag>;
    } else {
        // Check if the user is Staff type or Member type
        let verificationUrl = '';
        if (user.hasOwnProperty('roleId')) {
            verificationUrl = STAFF_URL + '/verification/' + user.id + '/' + verificationToken[0].token;
        } else {
            verificationUrl = MEMBER_URL + '/verification/' + user.id + '/' + verificationToken[0].token;
        }

        return (
            <div className="flex flex-row">
                <Tag color="orange">{t('verification-pending')}</Tag>
                <Tooltip title={t('copy-to-clipboard')}>
                    <CopyOutlined
                        className="hover:cursor-pointer"
                        onClick={() => {
                            copyToClipboard(verificationUrl);
                        }}
                    />
                </Tooltip>
            </div>
        );
    }
};

export default UserStatus;
