import { MouseEvent, useEffect, useState } from 'react';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { CloudUploadOutlined } from '@ant-design/icons';
import { Button, Upload } from 'antd';
import ImgCrop from 'antd-img-crop';
import { FILE_SIZE_LIMIT } from '@/configs';
import { uploadNoticeImage } from '@/services/site-setting';

interface DraggerProps {
    crop?: boolean;
    aspect?: number;
    value?: string;
    onChange?: (value: string) => void;
}

const Dragger: React.FC<DraggerProps> = ({ crop = true, aspect, value, onChange }) => {
    const { t } = useTranslation(['common']);
    const { Dragger } = Upload;
    const [url, setUrl] = useState<string | undefined>();

    useEffect(() => {
        setUrl(value);
    }, [value]);

    const onChangeHandler = (url: string) => {
        setUrl(url);
        onChange && onChange(url);
    };

    const onRemoveFile = (e: MouseEvent<HTMLElement>) => {
        e.preventDefault();
        e.stopPropagation();
        setUrl(undefined);
        onChange && onChange('');
    };

    return (
        // <ImgCrop
        //     fillColor="transparent"
        //     beforeCrop={crop ? undefined : () => false}
        //     onModalCancel={(resolve) => {
        //         resolve(Upload.LIST_IGNORE);
        //     }}
        // >
        <Dragger
            customRequest={(options) => uploadNoticeImage('/staff/site-setting/notice/upload', options)}
            accept="image/*"
            listType="picture"
            maxCount={1}
            showUploadList={false}
            onChange={(info) => {
                if (info.file.status === 'done') {
                    onChangeHandler(info.file.response.url);
                }
            }}
            beforeUpload={(file) => {
                const isLTSizeLimit = file.size / 1024 / 1024 > FILE_SIZE_LIMIT;

                if (isLTSizeLimit) {
                    toast.error(
                        t('messages:file-too-large', {
                            fileName: file.name,
                            size: FILE_SIZE_LIMIT,
                        }),
                    );
                    return Upload.LIST_IGNORE;
                }

                return true;
            }}
        >
            {url ? (
                <div>
                    <img className="mx-auto w-full max-w-[300px]" src={url} alt="image" />
                </div>
            ) : (
                <>
                    <p className="ant-upload-drag-icon">
                        <CloudUploadOutlined />
                    </p>
                    <p className="ant-upload-text">{t('common:only-one-image-allowed')}</p>
                    <p className="ant-upload-hint">{t('common:upload-image-to-replace-previous-one')}</p>
                </>
            )}
            {url ? (
                <Button danger onClick={onRemoveFile}>
                    {t('common:remove')}
                </Button>
            ) : (
                <p className="text-xs italic text-gray-500/80">{t('common:file-size-limit', { fileSizeLimit: '5 MB' })}</p>
            )}
        </Dragger>
        // </ImgCrop>
    );
};

export default Dragger;
