import React, { useState } from 'react';
import { useTranslation } from 'next-i18next';
import { FilterOutlined, SearchOutlined } from '@ant-design/icons';
import { Button, Col, DatePicker, Form, Input, Modal, Row, Select } from 'antd';
import { FormInstance } from 'antd/es/form/Form';

interface FilterAttributes {
    filterStudyGroupForm: FormInstance;
    onReset: () => void;
    onSearch: () => void;
    loading: boolean;
}

const { RangePicker } = DatePicker;

const FilterDrawer: React.FC<FilterAttributes> = ({ filterStudyGroupForm, onReset, onSearch, loading }) => {
    const { t } = useTranslation(['study-group', 'staff', 'common']);
    const criteriaSelected = filterStudyGroupForm.getFieldsValue();
    const criteriaCount: number = Object.values(criteriaSelected).reduce((count: number, val) => count + (val ? 1 : 0), 0);
    const [isModalOpen, setIsModalOpen] = useState(false);

    const breakPoint = {
        xs: 24,
        sm: 24,
        md: 12,
        lg: 12,
    };

    const onResetHandler = () => {
        onReset();
        setIsModalOpen(false);
    };

    const onSearchHandler = () => {
        onSearch();
        setIsModalOpen(false);
    };

    return (
        <div>
            <Button onClick={() => setIsModalOpen(true)} icon={<FilterOutlined />}>
                {criteriaCount > 0 && `(${criteriaCount})`} {t('common:filter')}
            </Button>
            <Modal title={t('common:filter')} onCancel={onResetHandler} open={isModalOpen} footer={null} width={650}>
                <Form form={filterStudyGroupForm} layout="vertical" name="filter_form" className="mt-6" size="large">
                    <Row gutter={[16, 0]}>
                        {/* Name */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="name" label={t('group-name')} labelCol={{ flex: '35px' }}>
                                <Input placeholder={t('enter-group-name')} />
                            </Form.Item>
                        </Col>
                        {/* Book Name */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="bookName" label={t('book-name')} labelCol={{ flex: '35px' }}>
                                <Input placeholder={t('enter-book-name')} />
                            </Form.Item>
                        </Col>
                    </Row>
                    <Row gutter={[16, 0]}>
                        {/* Start Date */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="startDate" label={t('start-date')} labelCol={{ flex: '35px' }}>
                                <DatePicker className="!w-full" />
                            </Form.Item>
                        </Col>
                        {/* End Date*/}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="endDate" label={t('end-date')} labelCol={{ flex: '35px' }}>
                                <DatePicker className="!w-full" />
                            </Form.Item>
                        </Col>
                    </Row>
                    <Row gutter={[16, 0]}>
                        {/* Created At */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="createdAt" label={t('common:created-at')} labelCol={{ flex: '35px' }}>
                                <RangePicker className="!w-full" />
                            </Form.Item>
                        </Col>
                        {/* Status */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="status" label={t('status')} labelCol={{ flex: '35px' }}>
                                <Select placeholder={t('common:select')}>
                                    <Select.Option value="">{t('select-a-status')}</Select.Option>
                                    <Select.Option value="ACTIVE">{t('common:active')}</Select.Option>
                                    <Select.Option value="ONGOING">{t('common:ongoing')}</Select.Option>
                                    <Select.Option value="INACTIVE">{t('common:inactive')}</Select.Option>
                                </Select>
                            </Form.Item>
                        </Col>
                    </Row>
                    <div className="flex justify-between mt-3 gap-x-3">
                        <Button onClick={onResetHandler} block loading={loading}>
                            {t('common:reset')}
                        </Button>
                        <Button type="primary" onClick={onSearchHandler} block loading={loading}>
                            {t('common:apply-filter')}
                            <SearchOutlined />
                        </Button>
                    </div>
                </Form>
            </Modal>
        </div>
    );
};

export default FilterDrawer;
