import { JSXElementConstructor, ReactElement, ReactNode, cloneElement, useState } from 'react';
import { useTranslation } from 'next-i18next';
import { Button, ButtonProps, Form, Input, Modal, Space } from 'antd';

interface ConfirmationModalProps {
    children: ReactElement<any, string | JSXElementConstructor<any>>;
    message: ReactNode | string;
    okButtonProps?: {
        type?: ButtonProps['type'];
        danger?: ButtonProps['danger'];
    };
    cancelText?: string;
    okText: string;
    onOk: (reason?: string) => void;
    reason?: boolean;
    width?: number;
    title?: string;
}

const ConfirmationModal: React.FC<ConfirmationModalProps> = ({
    children,
    message,
    okButtonProps,
    cancelText = 'cancel',
    okText,
    onOk,
    reason = false,
    width,
    title,
}) => {
    const { t } = useTranslation(['common']);
    const [reasonForm] = Form.useForm();

    const [modalOpen, setModalOpen] = useState<boolean>(false);

    const onCloseModalHandler = () => {
        setModalOpen(false);
        reasonForm.resetFields();
    };

    const onOkModalHandler = () => {
        if (reason) {
            reasonForm.validateFields().then((values) => {
                onOk(values.reason);
                setModalOpen(false);
                reasonForm.resetFields();
            });
        } else {
            onOk();
            setModalOpen(false);
        }
    };

    return (
        <>
            {cloneElement(children, { onClick: () => setModalOpen(true) })}
            <Modal
                width={width}
                open={modalOpen}
                maskClosable={true}
                footer={null}
                onCancel={onCloseModalHandler}
                closeIcon={false}
                centered
                title={title}
            >
                <Space direction="vertical" size="large" style={{ display: 'flex', gap: '4px' }}>
                    {reason ? (
                        <>
                            <Form form={reasonForm} layout="vertical" name="Reason Form" className="mt-2">
                                <Form.Item name="reason" rules={[{ required: true }]} label={message} labelCol={{ flex: '25px' }}>
                                    <Input.TextArea rows={3} placeholder={t('common:reason') as string} />
                                </Form.Item>
                            </Form>
                        </>
                    ) : (
                        <p className="mb-2">{message}</p>
                    )}
                    <div className="flex justify-end gap-2">
                        <Button block onClick={onCloseModalHandler}>
                            {cancelText}
                        </Button>
                        <Button block type="primary" onClick={onOkModalHandler} {...okButtonProps}>
                            {okText}
                        </Button>
                    </div>
                </Space>
            </Modal>
        </>
    );
};

export default ConfirmationModal;
