import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { Prisma, StudyGroupStatus } from '@prisma/client';
import * as dayjs from 'dayjs';
import { PrismaService } from 'src/prisma/prisma.service';
import { StudyGroupMembers, StudyGroupTaskCommentFilter, TotalPenaltyByStudyGroup } from 'src/types/study-group';
import { conditionalReturn } from 'src/utils';
import { CreateStudyGroupDto, ExportStudyGroupMembers, StudyGroupFilterParams, UpdateDescriptionDto, UpdateStudyGroupDto } from './study-group.dto';

@Injectable()
export class StudyGroupService {
    constructor(private prisma: PrismaService) {}

    // Context Provider
    async getStudyGroupById(studyGroupId: string) {
        const studyGroup = await this.prisma.studyGroup.findUnique({
            where: {
                id: studyGroupId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id', 'name', 'startDate', 'endDate', 'status', 'description']),
                book: {
                    select: {
                        ...this.prisma.createSelect(['id', 'name', 'link', 'uploadType']),
                        bookMedias: {
                            select: {
                                media: {
                                    select: {
                                        ...this.prisma.createSelect(['key', 'name']),
                                    },
                                },
                            },
                        },
                    },
                },
                studyGroupTask: {
                    orderBy: {
                        taskDate: 'asc',
                    },
                    select: {
                        ...this.prisma.createSelect(['id', 'taskDate']),
                        penalties: {
                            select: {
                                ...this.prisma.createSelect(['amount']),
                            },
                        },
                    },
                },
            },
        });

        if (!studyGroup) {
            throw new HttpException('api-messages:study-group-not-found', HttpStatus.NOT_FOUND);
        }

        return studyGroup;
    }

    async getStudyGroupMembers(studyGroupId: string) {
        const studyGroupMembers = await this.prisma.member.findMany({
            where: {
                deletedAt: null,
                studyGroupMembers: {
                    some: {
                        studyGroupId,
                    },
                },
            },
            select: {
                ...this.prisma.createSelect(['id', 'fullName', 'preferredName', 'phoneNumber', 'createdAt']),
                studyGroupMembers: {},
                medicalCertificate: {
                    select: this.prisma.exclude('MedicalCertificate', ['deletedAt']),
                },
            },
        });

        const memberCount = studyGroupMembers.length;

        return { studyGroupMembers, memberCount };
    }

    async getStudyGroupTaskById(studyGroupTaskId: string) {
        const studyGroupTask = await this.prisma.studyGroupTask.findFirst({
            where: {
                id: studyGroupTaskId,
            },
            select: {
                ...this.prisma.createSelect(['id', 'description', 'taskDate', 'status', 'studyGroupId']),
            },
        });

        if (!studyGroupTask) {
            throw new HttpException('api-messages:study-group-task-not-found', HttpStatus.NOT_FOUND);
        }

        const studyGroup = await this.prisma.studyGroup.findFirst({
            where: {
                id: studyGroupTask.studyGroupId,
            },
        });

        if (!studyGroup) {
            throw new HttpException('api-messages:study-group-not-found', HttpStatus.NOT_FOUND);
        }

        return studyGroupTask;
    }

    ///////////////////
    // Study Group
    ///////////////////

    // Get
    async getAllStudyGroupByPagination(query: StudyGroupFilterParams) {
        const { page, pageSize, sortField, sortOrder, name, bookName, status, startDate, endDate, createdAt, memberId } = query;

        const whereOptions: Prisma.StudyGroupWhereInput = {
            deletedAt: null,
            ...conditionalReturn(!!name, {
                name: { mode: 'insensitive', contains: name },
            }),
            ...conditionalReturn(!!bookName, {
                book: {
                    name: { mode: 'insensitive', contains: bookName },
                },
            }),
            ...conditionalReturn(!!status, {
                status,
            }),
            ...conditionalReturn(!!startDate, {
                startDate: {
                    gte: startDate && new Date(dayjs(startDate).startOf('day').toISOString()),
                    lte: startDate && new Date(dayjs(startDate).endOf('day').toISOString()),
                },
            }),
            ...conditionalReturn(!!endDate, {
                endDate: {
                    gte: endDate && new Date(dayjs(endDate).startOf('day').toISOString()),
                    lte: endDate && new Date(dayjs(endDate).endOf('day').toISOString()),
                },
            }),
            ...conditionalReturn(createdAt && createdAt.length === 2, {
                createdAt: {
                    gte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[0]).startOf('day').toISOString()),
                    lte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[1]).endOf('day').toISOString()),
                },
            }),
            ...conditionalReturn(!!memberId, {
                studyGroupMembers: {
                    some: {
                        memberId,
                    },
                },
            }),
        };

        const studyGroupCount = await this.prisma.studyGroup.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(studyGroupCount, page, pageSize);

        const studyGroupList = await this.prisma.studyGroup.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: {
                ...this.prisma.createSelect(['id', 'name', 'startDate', 'endDate', 'status', 'createdAt']),
                book: {
                    select: {
                        ...this.prisma.createSelect(['id', 'name']),
                    },
                },
                _count: {
                    select: {
                        studyGroupMembers: true,
                    },
                },
            },
        });

        return {
            total: studyGroupCount,
            rows: studyGroupList,
            page: currentPage,
        };
    }

    async getBooks() {
        return await this.prisma.book.findMany({
            where: {
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id', 'name']),
            },
        });
    }

    async getStudyGroupPenalty(studyGroupId: string) {
        const studyGroupTaskPenalty = await this.prisma.studyGroup.findMany({
            where: {
                id: studyGroupId,
            },
            select: {
                studyGroupTask: {
                    select: {
                        penalties: {
                            select: {
                                ...this.prisma.createSelect(['amount']),
                            },
                        },
                    },
                },
            },
        });

        const totalAmount = studyGroupTaskPenalty.reduce((acc, cur) => {
            const amount = cur.studyGroupTask.reduce((acc, cur) => {
                const amount = cur.penalties.reduce((acc, cur) => {
                    return acc + cur.amount;
                }, 0);
                return acc + amount;
            }, 0);
            return acc + amount;
        }, 0);

        return totalAmount;
    }

    async calculateAllPenaltiesByStudyGroup(studyGroupId: string) {
        const studyGroup = await this.prisma.studyGroup.findFirst({
            where: {
                id: studyGroupId,
                deletedAt: null,
            },
        });

        if (!studyGroup) {
            throw new HttpException('api-messages:study-group-not-found', HttpStatus.NOT_FOUND);
        }

        const penalties = await this.prisma.penalty.findMany({
            where: {
                studyGroupTask: {
                    studyGroupId: studyGroup.id,
                },
            },
            select: {
                id: true,
                member: {
                    select: {
                        id: true,
                        fullName: true,
                        phoneNumber: true,
                    },
                },
                amount: true,
                studyGroupTask: {
                    select: {
                        taskDate: true,
                    },
                },
            },
        });

        const calculatedPenalties: TotalPenaltyByStudyGroup[] = [];

        penalties.forEach((penalty) => {
            const { member, amount, studyGroupTask } = penalty;

            const index = calculatedPenalties.findIndex((item) => item.member.id === member.id);

            if (index === -1) {
                calculatedPenalties.push({
                    member: {
                        id: member.id,
                        fullName: member.fullName,
                        phoneNumber: member.phoneNumber,
                    },
                    amount,
                    studyGroupTask: {
                        taskDate: studyGroupTask.taskDate,
                    },
                });
            } else {
                calculatedPenalties[index].amount += amount;
            }
        });

        return calculatedPenalties;
    }

    async getMemberListByPagination(query: any, studyGroupId: string) {
        const { page, pageSize, sortField, sortOrder, fullName } = query;

        const memberWhereOptions: Prisma.MemberWhereInput = {
            deletedAt: null,
            NOT: {
                studyGroupMembers: {
                    some: {
                        studyGroupId,
                    },
                },
            },
        };

        if (fullName) {
            memberWhereOptions.fullName = {
                contains: fullName,
                mode: 'insensitive',
            };
        }

        const memberCount = await this.prisma.member.count({
            where: memberWhereOptions,
        });

        const currentPage = this.prisma.pageCounter(memberCount, page, pageSize);

        const memberList = await this.prisma.member.findMany({
            take: pageSize,
            skip: (currentPage - 1) * pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            where: memberWhereOptions,
            select: {
                ...this.prisma.createSelect(['id', 'fullName', 'preferredName', 'email']),
            },
        });

        return {
            count: memberCount,
            rows: memberList,
            page: currentPage,
        };
    }

    async getStudyGroupMemberList(studyGroupId: string, query: StudyGroupMembers) {
        const { page, pageSize, sortField, sortOrder, name, phoneNumber } = query;

        const whereOptions: Prisma.MemberWhereInput = {
            deletedAt: null,
            studyGroupMembers: {
                some: {
                    deletedAt: null,
                    studyGroupId,
                },
            },
            ...conditionalReturn(!!name, {
                OR: [
                    {
                        fullName: {
                            contains: name,
                            mode: 'insensitive',
                        },
                    },
                    {
                        preferredName: {
                            contains: name,
                            mode: 'insensitive',
                        },
                    },
                ],
            }),
            ...conditionalReturn(!!phoneNumber, {
                phoneNumber: {
                    contains: phoneNumber,
                    mode: 'insensitive',
                },
            }),
        };

        const memberCount = await this.prisma.member.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(memberCount, page, pageSize);

        const studyGroupMembers = await this.prisma.member.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: {
                ...this.prisma.createSelect(['id', 'fullName', 'preferredName', 'email', 'phoneNumber']),
            },
        });

        return {
            total: memberCount,
            rows: studyGroupMembers,
            page: currentPage,
        };
    }

    async getExportStudyGroupMemberList(query: ExportStudyGroupMembers, studyGroupId: string) {
        const { name } = query;

        const whereOptions: Prisma.StudyGroupMemberWhereInput = {
            deletedAt: null,
            studyGroupId,
            member: {
                deletedAt: null,
                ...conditionalReturn(!!name, {
                    OR: [
                        {
                            fullName: {
                                contains: name,
                                mode: 'insensitive',
                            },
                        },
                        {
                            preferredName: {
                                contains: name,
                                mode: 'insensitive',
                            },
                        },
                    ],
                }),
            },
        };

        const studyGroupMemberList = await this.prisma.studyGroupMember.findMany({
            where: whereOptions,
            select: {
                member: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName', 'preferredName', 'phoneNumber', 'email', 'createdAt']),
                    },
                },
            },
        });

        return studyGroupMemberList;
    }

    // Add
    async createStudyGroup(body: CreateStudyGroupDto) {
        const { name, bookId, groupPeriod } = body;

        if (dayjs(groupPeriod[0]).day() === dayjs().day()) {
            const studyGroup = await this.prisma.studyGroup.create({
                data: {
                    name,
                    bookId,
                    startDate: new Date(dayjs(groupPeriod[0]).startOf('day').toISOString()),
                    endDate: new Date(dayjs(groupPeriod[1]).endOf('day').toISOString()),
                    status: StudyGroupStatus.ONGOING,
                },
                select: {
                    ...this.prisma.createSelect(['id', 'name', 'startDate', 'endDate', 'status', 'createdAt']),
                },
            });

            // Generate Task for each day
            const duration = dayjs(studyGroup.endDate).diff(dayjs(studyGroup.startDate), 'day');

            await this.prisma.studyGroupTask.createMany({
                data: Array.from({ length: duration + 1 }).map((_, index) => ({
                    studyGroupId: studyGroup.id,
                    taskDate: new Date(dayjs(studyGroup.startDate).add(index, 'day').startOf('day').toISOString()),
                })),
            });

            return studyGroup;
        } else {
            const studyGroup = await this.prisma.studyGroup.create({
                data: {
                    name,
                    bookId,
                    startDate: new Date(dayjs(groupPeriod[0]).startOf('day').toISOString()),
                    endDate: new Date(dayjs(groupPeriod[1]).endOf('day').toISOString()),
                },
                select: {
                    ...this.prisma.createSelect(['id', 'name', 'startDate', 'endDate', 'status', 'createdAt']),
                },
            });

            return studyGroup;
        }
    }

    async createStudyGroupMember(studyGroupId: string, memberId: string) {
        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        const studyGroup = await this.prisma.studyGroup.findFirst({
            where: {
                id: studyGroupId,
                deletedAt: null,
            },
        });

        if (!studyGroup) {
            throw new HttpException('api-messages:study-group-not-found', HttpStatus.NOT_FOUND);
        }

        const studyGroupMember = await this.prisma.studyGroupMember.create({
            data: {
                studyGroupId,
                memberId,
            },
        });

        // Deduct mnember token
        await this.prisma.member.update({
            where: {
                id: memberId,
            },
            data: {
                bookTokens: member.bookTokens - 1,
            },
        });

        return studyGroupMember;
    }

    // Update
    async updateStudyGroupById(studyGroupId: string, body: UpdateStudyGroupDto) {
        const { name, groupPeriod, description } = body;

        const studyGroup = await this.prisma.studyGroup.findFirst({
            where: {
                id: studyGroupId,
                deletedAt: null,
            },
        });

        if (!studyGroup) {
            throw new HttpException('api-messages:study-group-not-found', HttpStatus.NOT_FOUND);
        }

        const updateStudyGroup = await this.prisma.studyGroup.update({
            where: {
                id: studyGroupId,
            },
            data: {
                name,
                startDate: new Date(dayjs(groupPeriod[0]).startOf('day').toISOString()),
                endDate: new Date(dayjs(groupPeriod[1]).endOf('day').toISOString()),
                description,
            },
            select: {
                ...this.prisma.createSelect(['id', 'name', 'status']),
            },
        });

        return updateStudyGroup;
    }

    // Delete
    async deleteStudyGroupById(studyGroupId: string) {
        const studyGroup = await this.prisma.studyGroup.findFirst({
            where: {
                id: studyGroupId,
                deletedAt: null,
            },
        });

        if (!studyGroup) {
            throw new HttpException('api-messages:study-group-not-found', HttpStatus.NOT_FOUND);
        }

        return await this.prisma.$transaction([
            this.prisma.studyGroup.update({
                where: {
                    id: studyGroupId,
                },
                data: {
                    deletedAt: new Date(),
                },
            }),
            this.prisma.studyGroupMember.updateMany({
                where: {
                    studyGroupId,
                },
                data: {
                    deletedAt: new Date(),
                },
            }),
        ]);
    }

    async deleteStudyGroupMemberById(studyGroupId: string, memberId: string) {
        const studyGroupMember = await this.prisma.studyGroupMember.findFirst({
            where: {
                studyGroupId,
                memberId,
            },
        });

        if (!studyGroupMember) {
            throw new HttpException('api-messages:study-group-member-not-found', HttpStatus.NOT_FOUND);
        }

        return await this.prisma.studyGroupMember.delete({
            where: {
                studyGroupId_memberId: {
                    studyGroupId,
                    memberId,
                },
            },
        });
    }

    ///////////////////
    // Study Group Task
    ///////////////////

    // Get
    async getTaskMemberPenalty(studyGroupTaskId: string) {
        const studyGroupTask = await this.prisma.studyGroupTask.findFirst({
            where: {
                id: studyGroupTaskId,
            },
        });

        if (!studyGroupTask) {
            throw new HttpException('api-messages:study-group-task-not-found', HttpStatus.NOT_FOUND);
        }

        const penalties = await this.prisma.penalty.findMany({
            where: {
                studyGroupTaskId,
            },
            select: {
                ...this.prisma.createSelect(['id', 'amount']),
                member: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName', 'preferredName']),
                    },
                },
            },
        });

        const totalAmount = penalties.reduce((acc, cur) => acc + cur.amount, 0);

        return { penalties, totalAmount };
    }

    async getTaskPenalty(studyGroupTaskId: string) {
        const studyGroupTaskPenalty = await this.prisma.studyGroupTask.findMany({
            where: {
                id: studyGroupTaskId,
            },
            select: {
                ...this.prisma.createSelect(['id']),
                penalties: {
                    select: {
                        ...this.prisma.createSelect(['amount']),
                    },
                },
            },
        });

        const totalAmount = studyGroupTaskPenalty.reduce((acc, cur) => {
            const amount = cur.penalties.reduce((acc, cur) => {
                return acc + cur.amount;
            }, 0);

            return acc + amount;
        }, 0);

        return { studyGroupTaskPenalty, totalAmount };
    }

    // Add
    async generateStudyGroupTask(studyGroupId: string) {
        const studyGroup = await this.prisma.studyGroup.findFirst({
            where: {
                id: studyGroupId,
                deletedAt: null,
            },
        });

        if (!studyGroup) {
            throw new HttpException('api-messages:study-group-not-found', HttpStatus.NOT_FOUND);
        }

        // Update study group status to generated
        await this.prisma.studyGroup.update({
            where: {
                id: studyGroupId,
            },
            data: {
                status: StudyGroupStatus.GENERATED,
            },
        });

        // Generate Task for each day
        const duration = dayjs(studyGroup.endDate).diff(dayjs(studyGroup.startDate), 'day');

        await this.prisma.studyGroupTask.createMany({
            data: Array.from({ length: duration + 1 }).map((_, index) => ({
                studyGroupId,
                taskDate: new Date(dayjs(studyGroup.startDate).add(index, 'day').startOf('day').toISOString()),
            })),
        });

        return studyGroup;
    }

    // Update
    async updateDescription(studyGroupTaskId: string, body: UpdateDescriptionDto) {
        const { description } = body;

        const studyGroupTask = await this.prisma.studyGroupTask.findFirst({
            where: {
                id: studyGroupTaskId,
                deletedAt: null,
            },
        });

        if (!studyGroupTask) {
            throw new HttpException('api-messages:study-group-task-not-found', HttpStatus.NOT_FOUND);
        }

        return await this.prisma.studyGroupTask.update({
            where: {
                id: studyGroupTaskId,
            },
            data: {
                description,
            },
        });
    }

    ///////////////////
    // Commment
    ///////////////////

    // Get
    async getTaskComments(studyGroupTaskId: string, query: StudyGroupTaskCommentFilter) {
        const { page, pageSize, name } = query;

        const whereOptions: Prisma.StudyGroupTaskCommentWhereInput = {
            studyGroupTaskId,
            deletedAt: null,
            prePlanned: false,
            ...conditionalReturn(!!name, {
                member: {
                    OR: [
                        {
                            fullName: {
                                contains: name,
                                mode: 'insensitive',
                            },
                        },
                        {
                            preferredName: {
                                contains: name,
                                mode: 'insensitive',
                            },
                        },
                    ],
                },
            }),
        };

        const studyGroupTaskCommentCount = await this.prisma.studyGroupTaskComment.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(studyGroupTaskCommentCount, page, pageSize);

        const studyGroupTaskComments = await this.prisma.studyGroupTaskComment.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                commentDate: 'desc',
            },
            select: {
                ...this.prisma.createSelect(['id', 'comment', 'commentDate', 'createdAt', 'updatedAt', 'prePlanned', 'prePlannedDate', 'parent_id']),
                member: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName', 'preferredName']),
                    },
                },
                staff: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName']),
                    },
                },
                Children: {
                    select: {
                        id: true,
                    }
                }
            },
        });

        return {
            total: studyGroupTaskCommentCount,
            rows: studyGroupTaskComments,
            page: currentPage,
        };
    }

    async getCommentsByGroupId(studyGroupId: string) {
        const studyGroupTaskComments = await this.prisma.studyGroupTaskComment.findMany({
            where: {
                studyGroupTask: {
                    studyGroupId,
                },
                deletedAt: null,
            },
            orderBy: {
                studyGroupTask: {
                    taskDate: 'asc',
                },
            },
            select: {
                ...this.prisma.createSelect(['id', 'comment']),
                member: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName', 'preferredName']),
                    },
                },
                studyGroupTask: {
                    select: {
                        ...this.prisma.createSelect(['id', 'taskDate']),
                    },
                },
            },
        });

        return studyGroupTaskComments;
    }

    async getTaskCommentLikes(studyGroupTaskId: string) {
        const studyGroupTaskCommentLikes = await this.prisma.studyGroupTaskCommentLike.findMany({
            where: {
                studyGroupTaskId,
            },
            select: {
                ...this.prisma.createSelect(['id', 'studyGroupTaskId', 'studyGroupTaskCommentId', 'staffId', 'memberId']),
            },
        });

        return studyGroupTaskCommentLikes;
    }

    private async getNestedReplies(studyGroupTaskId: string, parentId: string) {
        const replies = await this.prisma.studyGroupTaskComment.findMany({
            where: {
                studyGroupTaskId,
                parent_id: parentId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id', 'comment', 'studyGroupTaskId', 'commentDate', 'prePlanned', 'prePlannedDate', 'parent_id']),
                staff: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName']),
                    },
                },
                member: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName', 'preferredName']),
                    },
                },
            },
        });

        // Recursively fetch nested replies
        const nestedReplies = await Promise.all(
            replies.map(async (reply) => {
                const children = await this.getNestedReplies(studyGroupTaskId, reply.id);
                return {
                    ...reply,
                    Children: children,
                };
            }),
        );

        return nestedReplies;
    }

    async getReplies(studyGroupTaskId: string, parentId: string) {
        const studyGroupTaskCommentReplies = await this.getNestedReplies(studyGroupTaskId, parentId);
        return studyGroupTaskCommentReplies;
    }

    async getOwnTaskComment(studyGroupTaskId: string, staffId: string) {
        const studyGroupTaskComments = await this.prisma.studyGroupTaskComment.findFirst({
            where: {
                studyGroupTaskId,
                staffId,
                parent_id: null,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id', 'comment', 'commentDate', 'createdAt', 'updatedAt', 'prePlanned', 'prePlannedDate', 'parent_id']),
                staff: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName']),
                    },
                },
                member: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName', 'preferredName']),
                    },
                },
                studyGroupTask: {
                    select: {
                        ...this.prisma.createSelect(['id']),
                    },
                },
            },
        });

        return studyGroupTaskComments;
    }

    // Comments by id for members who have not commented
    async getCommentsByTaskId(studyGroupTaskId: string) {
        const studyGroupTaskComments = await this.prisma.studyGroupTaskComment.findMany({
            where: {
                studyGroupTaskId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id', 'studyGroupTaskId', 'memberId']),
            },
        });

        return studyGroupTaskComments;
    }

    // Delete
    async deleteStudyGroupSubTaskComment(commentId: string) {
        const studyGroupSubTaskComment = await this.prisma.studyGroupTaskComment.findFirst({
            where: {
                id: commentId,
                deletedAt: null,
            },
        });

        if (!studyGroupSubTaskComment) {
            throw new HttpException('api-messages:study-group-sub-task-comment-not-found', HttpStatus.NOT_FOUND);
        }

        return await this.prisma.studyGroupTaskComment.update({
            where: {
                id: commentId,
            },
            data: {
                deletedAt: new Date(),
            },
            select: {
                id: true,
                studyGroupTask: {
                    select: {
                        id: true,
                    },
                },
            },
        });
    }

    // Update
    async likeTaskComment(studyGroupTaskCommentId: string, staffId: string, studyGroupTaskId: string) {
        const studyGroupTaskComment = await this.prisma.studyGroupTaskComment.findFirst({
            where: {
                id: studyGroupTaskCommentId,
                deletedAt: null,
            },
        });

        if (!studyGroupTaskComment) {
            throw new HttpException('api-messages:study-group-task-comment-not-found', HttpStatus.NOT_FOUND);
        }

        let studyGroupTaskCommentLike = await this.prisma.studyGroupTaskCommentLike.findFirst({
            where: {
                studyGroupTaskCommentId,
                staffId,
            },
        });

        if (studyGroupTaskCommentLike) {
            await this.prisma.studyGroupTaskCommentLike.delete({
                where: {
                    id: studyGroupTaskCommentLike.id,
                },
            });
            studyGroupTaskCommentLike = null;
        } else {
            studyGroupTaskCommentLike = await this.prisma.studyGroupTaskCommentLike.create({
                data: {
                    staffId,
                    studyGroupTaskCommentId,
                    studyGroupTaskId,
                },
            });
        }

        return studyGroupTaskCommentLike;
    }

    async updateReply(studyGroupTaskId: string, staffId: string, commentId: string, reply: string) {
        const studyGroupTaskComment = await this.prisma.studyGroupTaskComment.findFirst({
            where: {
                id: commentId,
                studyGroupTaskId,
                deletedAt: null,
            },
        });

        if (!studyGroupTaskComment) {
            throw new HttpException('api-messages:study-group-task-comment-not-found', HttpStatus.NOT_FOUND);
        }

        const memberStudyGroupTaskReply = await this.prisma.studyGroupTaskComment.findFirst({
            where: {
                staffId,
                studyGroupTaskId,
                parent_id: commentId,
                deletedAt: null,
            },
        });

        if (memberStudyGroupTaskReply) {
            const updatedReply = await this.prisma.studyGroupTaskComment.update({
                where: {
                    id: memberStudyGroupTaskReply.id,
                    deletedAt: null,
                },
                data: {
                    comment: reply,
                    commentDate: new Date(),
                },
            });

            return updatedReply;
        } else {
            const createdReply = await this.prisma.studyGroupTaskComment.create({
                data: {
                    comment: reply,
                    staffId,
                    studyGroupTaskId,
                    commentDate: new Date(),
                    parent_id: commentId,
                },
            });

            return createdReply;
        }
    }
}
