import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { PrismaService } from 'src/prisma/prisma.service';
import { CustomException } from 'src/utils/CustomException';
import { v4 as uuidv4 } from 'uuid';
import * as dayjs from 'dayjs';
import { EmailService } from 'src/email/email.service';
import { StaffQuery } from 'src/types';
import { Prisma } from '@prisma/client';
import staffData from 'src/data/staff';
import { CreateStaffDto, UpdateStaffEmailDto, UpdateStaffPhoneNumberDto, UpdateStaffProfileDto } from './staff.dto';
import { StaffSharedService } from '../staff-shared.service';
import * as argon from 'argon2';
@Injectable()
export class StaffService {
    constructor(
        private prisma: PrismaService,
        private email: EmailService,
        private shared: StaffSharedService,
    ) {}

    async getStaff(query: StaffQuery) {
        const { page, pageSize, sortField, sortOrder, fullNameEmail, role } = query;

        // Find staff where params
        const staffWhereOptions: Prisma.StaffWhereInput = {
            deletedAt: null,
        };

        // Search by full name or email
        if (fullNameEmail) {
            staffWhereOptions.OR = [
                {
                    fullName: {
                        contains: fullNameEmail,
                        mode: 'insensitive',
                    },
                },
                {
                    email: {
                        contains: fullNameEmail,
                        mode: 'insensitive',
                    },
                },
            ];
        }

        // Search by role
        if (role) {
            staffWhereOptions.roleId = role;
        }

        const staffCount = await this.prisma.staff.count({
            where: staffWhereOptions,
        });

        // If page is out of range, return the last page
        const currentPage = this.prisma.pageCounter(staffCount, page, pageSize);

        // Get filtered staff and filtered staff count
        const staff = await this.prisma.staff.findMany({
            take: pageSize,
            skip: (currentPage - 1) * pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            where: staffWhereOptions,
            select: {
                ...this.prisma.createSelect(staffData.exclude(['deletedAt'])),
                tokens: {
                    select: {
                        ...this.prisma.createSelect(['type', 'token', 'expiredAt', 'usedAt']),
                    },
                    orderBy: {
                        createdAt: 'desc',
                    },
                    take: 1,
                },
            },
        });

        // Map staff password to true if password exists, false if not
        const mappedStaff = staff.map((item) => {
            const hasPassword = !!item.password;
            delete item.password;
            return {
                ...item,
                password: hasPassword,
            };
        });

        return {
            count: staffCount,
            rows: mappedStaff,
            page: currentPage,
        };
    }

    async getStaffById(staffId: string) {
        const staff = await this.prisma.staff.findFirst({
            where: {
                id: staffId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(staffData.exclude(['deletedAt'])),
                role: {
                    select: this.prisma.createSelect(['id', 'name']),
                },
                tokens: {
                    select: {
                        ...this.prisma.createSelect(['type', 'token', 'expiredAt', 'usedAt']),
                    },
                    orderBy: {
                        createdAt: 'desc',
                    },
                    take: 1,
                },
            },
        });

        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        // Map staff password to true if password exists, false if not
        const hasPassword = !!staff.password;
        delete staff.password;

        return {
            ...staff,
            password: hasPassword,
        };
    }

    async createStaff(body: CreateStaffDto) {
        // Check existing email
        const { isCreated } = await this.shared.emailCheck(body.email);

        // If email already exists, throw error
        if (isCreated) {
            throw new HttpException('api-messages:email-already-exists', HttpStatus.BAD_REQUEST);
        }

        // Create verification token
        const token = uuidv4();

        return this.prisma.$transaction(async (tx) => {
            // Create staff
            const staff = await tx.staff.create({
                data: {
                    email: body.email,
                    fullName: body.fullName,
                    roleId: body.roleId,
                    phoneNumber: body.phoneNumber,
                    lastActiveAt: dayjs().toDate(),
                    tokens: {
                        create: {
                            type: 'CONFIRMATION',
                            token,
                            createdAt: dayjs().toDate(),
                            expiredAt: dayjs().add(3, 'days').toDate(),
                        },
                    },
                },
                select: {
                    ...this.prisma.createSelect(['id', 'email', 'fullName', 'roleId']),
                },
            });

            // Send verification email
            const emailResponse = await this.email.staffVerification(staff.email, {
                name: staff.fullName,
                staffId: staff.id,
                token,
            });

            // If email failed to send, throw error
            if (!emailResponse.success) {
                throw new HttpException('api-messages:email-failed-to-send', HttpStatus.INTERNAL_SERVER_ERROR);
            }

            return staff;
        });
    }

    async updateStaffStatus(staffId: string) {
        const staff = await this.prisma.staff.findFirst({
            where: {
                id: staffId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
                status: true,
            },
        });

        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        const updatedStaff = await this.prisma.staff.update({
            where: {
                id: staff.id,
            },
            data: {
                status: staff.status === 'ACTIVE' ? 'INACTIVE' : 'ACTIVE',
            },
            select: this.prisma.createSelect(staffData.exclude(['password', 'deletedAt'])),
        });

        return updatedStaff;
    }

    async updateStaffRole(myStaffId: string, staffId: string, roleId: string) {
        const staff = await this.prisma.staff.findFirst({
            where: {
                id: staffId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
                role: {
                    select: this.prisma.createSelect(['superAdmin']),
                },
            },
        });

        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        const myDetail = await this.prisma.staff.findFirst({
            where: {
                id: myStaffId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
                role: {
                    select: this.prisma.createSelect(['superAdmin']),
                },
            },
        });

        if (!myDetail) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        if (staff.role['superAdmin'] && !myDetail.role['superAdmin']) {
            throw new HttpException('api-messages:permission-denied', HttpStatus.UNAUTHORIZED);
        }

        const updatedStaff = await this.prisma.staff.update({
            where: {
                id: staff.id,
            },
            data: {
                roleId,
            },
            select: this.prisma.createSelect(staffData.exclude(['password', 'deletedAt'])),
        });

        return updatedStaff;
    }

    async updateStaffProfile(staffId: string, body: UpdateStaffProfileDto) {
        const staff = await this.prisma.staff.findFirst({
            where: {
                id: staffId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        const updatedStaff = await this.prisma.staff.update({
            where: {
                id: staff.id,
            },
            data: {
                fullName: body.fullName,
            },
            select: this.prisma.createSelect(staffData.exclude(['password', 'deletedAt'])),
        });

        return updatedStaff;
    }

    async updateStaffEmail(staffId: string, body: UpdateStaffEmailDto) {
        const staff = await this.prisma.staff.findFirst({
            where: {
                id: staffId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        // Check existing email
        const { isCreated } = await this.shared.emailCheck(body.email, staffId);

        // If email already exists, throw error
        if (isCreated) {
            throw new HttpException('api-messages:email-already-exists', HttpStatus.BAD_REQUEST);
        }

        const updatedStaff = await this.prisma.staff.update({
            where: {
                id: staff.id,
            },
            data: {
                email: body.email,
            },
            select: this.prisma.createSelect(staffData.exclude(['password', 'deletedAt'])),
        });

        return updatedStaff;
    }

    async updateStaffPhoneNumber(staffId: string, body: UpdateStaffPhoneNumberDto) {
        const staff = await this.prisma.staff.findFirst({
            where: {
                id: staffId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        const updatedStaff = await this.prisma.staff.update({
            where: {
                id: staff.id,
            },
            data: {
                phoneNumber: body.phoneNumber,
            },
            select: this.prisma.createSelect(staffData.exclude(['password', 'deletedAt'])),
        });

        return updatedStaff;
    }

    async updateStaffPassword(myStaffId: string, staffId: string, password: string) {
        const staff = await this.prisma.staff.findFirst({
            where: {
                id: staffId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
                role: {
                    select: this.prisma.createSelect(['superAdmin']),
                },
            },
        });

        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        const myDetail = await this.prisma.staff.findFirst({
            where: {
                id: myStaffId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
                role: {
                    select: this.prisma.createSelect(['superAdmin']),
                },
            },
        });

        if (!myDetail) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        if (staff.role['superAdmin'] && !myDetail.role['superAdmin']) {
            throw new HttpException('api-messages:permission-denied', HttpStatus.UNAUTHORIZED);
        }

        // Hash password
        const hashedPassword = await argon.hash(password);

        // Update staff
        const updatedStaff = await this.prisma.staff.update({
            where: {
                id: staff.id,
            },
            data: {
                password: hashedPassword,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
            },
        });

        return updatedStaff;
    }

    async deleteStaff(myStaffId: string, staffId: string) {
        // If staff operate delete himself, throw error
        if (myStaffId === staffId) {
            throw new HttpException('api-messages:cannot-delete-yourself', HttpStatus.BAD_REQUEST);
        }

        // Find staff
        const staff = await this.prisma.staff.findFirst({
            where: {
                id: staffId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
                role: {
                    select: this.prisma.createSelect(['superAdmin']),
                },
            },
        });

        // If staff not found, throw error
        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        // Find my detail
        const myDetail = await this.prisma.staff.findFirst({
            where: {
                id: staffId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
                role: {
                    select: this.prisma.createSelect(['superAdmin']),
                },
            },
        });

        // If my detail not found, throw error
        if (!myDetail) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        // If staff is super admin and my detail is not super admin, throw error
        if (staff.role['superAdmin'] && !myDetail.role['superAdmin']) {
            throw new HttpException('api-messages:permission-denied', HttpStatus.UNAUTHORIZED);
        }

        // Soft delete staff
        const deletedStaff = await this.prisma.staff.update({
            where: {
                id: staff.id,
            },
            data: {
                deletedAt: dayjs().toDate(),
                status: 'INACTIVE',
            },
            select: this.prisma.createSelect(staffData.exclude(['password', 'deletedAt'])),
        });

        return deletedStaff;
    }

    async sendVerificationEmail(staffId: string, email: string, fullName: string) {
        // Ensure the verification request is not more than 5 times per 12 hours
        const staffTokens = await this.prisma.staffToken.findMany({
            where: {
                staffId,
                type: 'CONFIRMATION',
                createdAt: {
                    gte: dayjs().subtract(12, 'hours').toDate(),
                },
            },
        });

        if (staffTokens.length >= 5) {
            throw new CustomException('api-messages:too-many-verification-requests', HttpStatus.TOO_MANY_REQUESTS, {
                tooManyRequests: true,
            });
        }

        // Create new token
        const newToken = uuidv4();
        const newTokenExpiredAt = dayjs().add(3, 'days');

        await this.prisma.staffToken.create({
            data: {
                staffId,
                token: newToken,
                type: 'CONFIRMATION',
                expiredAt: newTokenExpiredAt.toDate(),
            },
        });

        // Send verification email
        const emailResponse = await this.email.staffVerification(email, {
            name: fullName,
            staffId,
            token: newToken,
        });

        return emailResponse;
    }

    async resendVerificationEmail(email: string) {
        const { isCreated, isVerified } = await this.emailCheck(email);

        // Ensure staff is created but not yet verified
        if (!isCreated) {
            throw new CustomException('api-messages:staff-not-found', HttpStatus.NOT_FOUND, {
                isCreated,
            });
        }
        if (isVerified) {
            throw new CustomException('api-messages:staff-already-verified', HttpStatus.CONFLICT, {
                isCreated,
                isVerified,
            });
        }

        // Resend verification email
        const staff = await this.prisma.staff.findFirst({
            where: {
                email,
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
            },
        });

        const emailResponse = await this.sendVerificationEmail(staff.id, staff.email, staff.fullName);

        // If email failed to send, throw error
        if (!emailResponse.success) {
            throw new HttpException('api-messages:email-failed-to-send', HttpStatus.INTERNAL_SERVER_ERROR);
        }

        return staff;
    }

    private async emailCheck(
        email: string,
        excludedId?: string,
    ): Promise<{
        isCreated: boolean;
        isVerified: boolean;
    }> {
        const staff = await this.prisma.staff.findFirst({
            where: {
                email,
                deletedAt: null,
                id: !!excludedId
                    ? {
                          not: excludedId,
                      }
                    : undefined,
            },
        });

        if (!staff) {
            // Staff does not exist
            return {
                isCreated: false,
                isVerified: false,
            };
        }

        if (!staff.password) {
            // Staff exists but not verified
            return {
                isCreated: true,
                isVerified: false,
            };
        }

        // Staff exists and verified
        return {
            isCreated: true,
            isVerified: true,
        };
    }
}
