import { Body, Controller, Delete, Get, Param, Post, Put, Query, UploadedFile, UseInterceptors } from '@nestjs/common';
import { SiteSettingService } from './site-setting.service';
import { Authenticator, Authorizer } from 'src/auth/auth.decorator';
import { User } from 'src/types';
import { PackageDto, PackageQuery, PenaltyDto, UpdateCompanyInformationDto } from './site-setting.dto';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { addPackageSchema, packageQuerySchema } from './site-setting.validator';
import { FileInterceptor } from '@nestjs/platform-express';

@Controller()
export class SiteSettingController {
    constructor(private siteSetting: SiteSettingService) {}

    @Authenticator(User.STAFF)
    @Authorizer('SITE_SETTING')
    @Get('package')
    async getPackageList(@Query(new JoiValidationPipe(packageQuerySchema)) query: PackageQuery) {
        const response = await this.siteSetting.getPackages(query);

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('SITE_SETTING')
    @Post('package')
    async addPackage(@Body(new JoiValidationPipe(addPackageSchema)) body: PackageDto) {
        const response = await this.siteSetting.addPackage(body);
        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('SITE_SETTING')
    @Put('package/:packageId')
    async updatePackage(@Body() body: PackageDto, @Param('packageId') packageId: string) {
        const response = await this.siteSetting.updatePackage(packageId, body);
        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('SITE_SETTING')
    @Delete('package/:packageId')
    async deletePackage(@Param('packageId') packageId: string) {
        const response = await this.siteSetting.deletePackage(packageId);
        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('SITE_SETTING')
    @Get('penalty-amount')
    async getPenaltyAmount() {
        const response = await this.siteSetting.getPenaltyAmount();

        return response;
    }
    @Authenticator(User.STAFF)
    @Authorizer('SITE_SETTING')
    @Put('penalty-amount')
    async updatePenaltyAmount(@Body() body: PenaltyDto) {
        const response = await this.siteSetting.updatePenaltyAmount(body);

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('SITE_SETTING')
    @Get('company-information')
    async getCompanyInformation() {
        const response = await this.siteSetting.getCompanyInformation();

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('SITE_SETTING')
    @Put('company-information')
    async updateCompanyInformation(@Body() body: UpdateCompanyInformationDto) {
        const response = await this.siteSetting.updateCompanyInformation(body);

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('SITE_SETTING')
    @Post('notice/upload')
    @UseInterceptors(FileInterceptor('file'))
    async uploadNoticeImage(@UploadedFile() file: Express.Multer.File) {
        const response = await this.siteSetting.uploadNoticeImage(file);

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('SITE_SETTING')
    @Get('notice')
    async getNotices() {
        const response = await this.siteSetting.getNotices();

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('SITE_SETTING')
    @Put('notice')
    async updateNotices(@Body('notices') notices: any) {
        const response = await this.siteSetting.updateNotices(notices);

        return response;
    }
}
