import { Body, Controller, Delete, Get, Param, Post, Put, Query, Res } from '@nestjs/common';
import { Authenticator, Authorizer, Staff } from 'src/auth/auth.decorator';
import { RolesQuery, StaffInRequest, User } from 'src/types';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { CreateRoleDto, UpdateRoleDto } from './role.dto';
import { RoleService } from './role.service';
import { createRoleSchema, rolesQuerySchema, updateRoleSchema } from './role.validator';
import { ActivityLogService } from 'src/activity-log/activity-log.service';

@Authenticator(User.STAFF)
@Controller()
export class RoleController {
    constructor(
        private roleService: RoleService,
        private activityLogService: ActivityLogService,
    ) {}

    @Authenticator(User.STAFF)
    @Authorizer('ROLE_VIEW')
    @Get('')
    async getRoles(@Query(new JoiValidationPipe(rolesQuerySchema)) query: RolesQuery) {
        const roleList = await this.roleService.getRoleList(query);
        return roleList;
    }

    @Authenticator(User.STAFF)
    @Authorizer('ROLE_VIEW')
    @Get(':roleId')
    async getSingleRole(@Param('roleId') roleId: string) {
        return await this.roleService.getSingleRole(roleId);
    }

    @Authenticator(User.STAFF)
    @Authorizer('ROLE_CREATE')
    @Post('')
    async createRole(@Staff() staff: StaffInRequest, @Body(new JoiValidationPipe(createRoleSchema)) body: CreateRoleDto) {
        const { id: staffId, fullName } = staff;
        const response = await this.roleService.createRole(body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'role',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'CREATE',
            description: 'api-messages:create-role',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('ROLE_UPDATE')
    @Put(':roleId')
    async updateRole(
        @Staff() staff: StaffInRequest,
        @Param('roleId') roleId: string,
        @Body(new JoiValidationPipe(updateRoleSchema))
        body: UpdateRoleDto,
    ) {
        const { id: staffId, fullName } = staff;
        const response = await this.roleService.updateRole(roleId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'role',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-role',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('ROLE_DELETE')
    @Delete(':roleId')
    async deleteRole(@Staff() staff: StaffInRequest, @Param('roleId') roleId: string) {
        const { id: staffId, fullName } = staff;
        const response = await this.roleService.deleteRole(roleId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'role',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'DELETE',
            description: 'api-messages:delete-role',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('ROLE_UPDATE')
    @Put('')
    async restoreRole(@Staff() staff: StaffInRequest, @Body() body: { roleId: string }) {
        const { id: staffId, fullName } = staff;
        const response = await this.roleService.restoreRole(body.roleId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'role',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'RESTORE',
            description: 'api-messages:restore-role',
            data: response,
        });

        return response;
    }
}
