import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { BookTokenSubscriptionStatus, Prisma } from '@prisma/client';
import { MediaService } from 'src/media/media.service';
import { PrismaService } from 'src/prisma/prisma.service';
import { BookTokenParams, BookTokenQuery, UpdateBookTokenDto } from './book-token.dto';
import { conditionalReturn } from 'src/utils';
import * as dayjs from 'dayjs';

@Injectable()
export class StaffBookTokenService {
    constructor(
        private prisma: PrismaService,
        private readonly media: MediaService,
    ) {}

    async getBookTokenSubscriptionList(query: BookTokenQuery) {
        const { page, pageSize, sortField, sortOrder, refNo, memberName, status, packageName, paymentDate, createdAt } = query;

        const whereOptions: Prisma.BookTokenSubscriptionWhereInput = {
            deletedAt: null,
            ...conditionalReturn(!!refNo, {
                displayId: { mode: 'insensitive', contains: refNo },
            }),
            ...conditionalReturn(!!memberName, {
                member: {
                    fullName: { mode: 'insensitive', contains: memberName },
                },
            }),
            ...conditionalReturn(!!status, { paymentStatus: status }),
            ...conditionalReturn(paymentDate && paymentDate.length === 2, {
                paymentDate: {
                    gte: paymentDate && paymentDate.length === 2 && new Date(dayjs(paymentDate[0]).startOf('day').toISOString()),
                    lte: paymentDate && paymentDate.length === 2 && new Date(dayjs(paymentDate[1]).endOf('day').toISOString()),
                },
            }),
            ...conditionalReturn(createdAt && createdAt.length === 2, {
                createdAt: {
                    gte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[0]).startOf('day').toISOString()),
                    lte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[1]).endOf('day').toISOString()),
                },
            }),
            ...conditionalReturn(!!packageName, {
                package: {
                    name: { mode: 'insensitive', contains: packageName },
                },
            }),
        };

        const subscriptionCount = await this.prisma.bookTokenSubscription.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(subscriptionCount, page, pageSize);

        const bookTokenSubscriptionList = await this.prisma.bookTokenSubscription.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: {
                ...this.prisma.createSelect([
                    'id',
                    'price',
                    'displayId',
                    'paymentStatus',
                    'reason',
                    'paymentDate',
                    'quantity',
                    'numberOfBookTokens',
                    'createdAt',
                    'expiredAt',
                ]),
                member: {
                    select: this.prisma.createSelect(['id', 'fullName', 'preferredName']),
                },
                package: {
                    select: this.prisma.createSelect(['id', 'name', 'price', 'numberOfBookToken', 'description']),
                },
            },
        });

        return {
            total: subscriptionCount,
            rows: bookTokenSubscriptionList,
            page: currentPage,
        };
    }

    async getBookTokenSubscriptionById(bookTokenId: string) {
        const bookTokenSubscription = await this.prisma.bookTokenSubscription.findUnique({
            where: {
                id: bookTokenId,
            },
            select: {
                ...this.prisma.createSelect(['id', 'price', 'displayId', 'paymentStatus', 'reason', 'quantity', 'expiredAt']),
                bookTokenSubscriptionMedias: {
                    select: {
                        media: {
                            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
                        },
                    },
                },
                member: {
                    select: this.prisma.createSelect(['id', 'fullName', 'preferredName']),
                },
                package: {
                    select: this.prisma.createSelect(['name', 'description']),
                },
            },
        });

        if (!bookTokenSubscription) {
            throw new HttpException('api-messages:book-token-not-found', HttpStatus.NOT_FOUND);
        }

        return bookTokenSubscription;
    }

    async getExportBookTokenSubscriptionList(query: BookTokenParams) {
        const { refNo, memberName, status, packageName, paymentDate, createdAt } = query;

        const whereOptions: Prisma.BookTokenSubscriptionWhereInput = {
            deletedAt: null,
            ...conditionalReturn(!!refNo, {
                displayId: { mode: 'insensitive', contains: refNo },
            }),
            ...conditionalReturn(!!memberName, {
                member: {
                    fullName: { mode: 'insensitive', contains: memberName },
                },
            }),
            ...conditionalReturn(!!status, { paymentStatus: status }),
            ...conditionalReturn(paymentDate && paymentDate.length === 2, {
                paymentDate: {
                    gte: paymentDate && paymentDate.length === 2 && new Date(dayjs(paymentDate[0]).startOf('day').toISOString()),
                    lte: paymentDate && paymentDate.length === 2 && new Date(dayjs(paymentDate[1]).endOf('day').toISOString()),
                },
            }),
            ...conditionalReturn(createdAt && createdAt.length === 2, {
                createdAt: {
                    gte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[0]).startOf('day').toISOString()),
                    lte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[1]).endOf('day').toISOString()),
                },
            }),
            ...conditionalReturn(!!packageName, {
                package: {
                    name: { mode: 'insensitive', contains: packageName },
                },
            }),
        };

        const bookTokenSubscriptionList = await this.prisma.bookTokenSubscription.findMany({
            where: whereOptions,
            select: {
                package: {
                    select: this.prisma.createSelect(['name', 'description']),
                },
                member: {
                    select: this.prisma.createSelect(['fullName', 'phoneNumber', 'address']),
                },
                ...this.prisma.createSelect(['price', 'paymentStatus', 'paymentDate', 'quantity', 'displayId', 'createdAt']),
            },
        });

        return bookTokenSubscriptionList;
    }

    async uploadPaymentProofMedia(file: Express.Multer.File) {
        const fileResponse = await this.media.upload(file, { isPublic: false });

        const mediaResponse = await this.prisma.media.create({
            data: fileResponse,
            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
        });

        return mediaResponse;
    }

    async updateBookTokenSubscriptionById(bookTokenId: string, body: UpdateBookTokenDto) {
        const isBookTokenSubscriptionExist = await this.prisma.bookTokenSubscription.findUnique({
            where: {
                id: bookTokenId,
            },
            select: {
                id: true,
                memberId: true,
                package: {
                    select: this.prisma.createSelect(['numberOfBookToken']),
                },
                paymentStatus: true,
            },
        });

        if (!isBookTokenSubscriptionExist) {
            throw new HttpException('api-messages:book-token-not-found', HttpStatus.NOT_FOUND);
        }

        const bookToken = await this.prisma.bookTokenSubscription.update({
            where: {
                id: bookTokenId,
            },
            data: {
                quantity: body.quantity,
                numberOfBookTokens: body.quantity * isBookTokenSubscriptionExist.package.numberOfBookToken,
                paymentStatus: body.paymentStatus,
                reason: body.paymentStatus === BookTokenSubscriptionStatus.REJECTED ? body.reason : '',
                expiredAt: body.paymentStatus === BookTokenSubscriptionStatus.PAID ? body.expiredAt : null,
                bookTokenSubscriptionMedias: {
                    deleteMany: {},
                    createMany: {
                        data: body.uploadFile.map((data) => ({
                            mediaId: data.mediaId,
                        })),
                    },
                },
            },
            select: {
                id: true,
                memberId: true,
                package: {
                    select: this.prisma.createSelect(['numberOfBookToken']),
                },
            },
        });

        /* If paymentStatus === PAID then update book token value in Member table*/
        if (
            isBookTokenSubscriptionExist.paymentStatus !== BookTokenSubscriptionStatus.PAID &&
            body.paymentStatus === BookTokenSubscriptionStatus.PAID
        ) {
            await this.prisma.member.update({
                where: {
                    id: bookToken.memberId,
                },
                data: {
                    // Add on the current book token value
                    bookTokens: {
                        increment: body.quantity * bookToken.package.numberOfBookToken,
                    },
                },
                select: { id: true },
            });
        }

        if (
            isBookTokenSubscriptionExist.paymentStatus === BookTokenSubscriptionStatus.PAID &&
            body.paymentStatus !== BookTokenSubscriptionStatus.PAID
        ) {
            await this.prisma.member.update({
                where: {
                    id: bookToken.memberId,
                },
                data: {
                    // Add on the current book token value
                    bookTokens: {
                        decrement: body.quantity * bookToken.package.numberOfBookToken,
                    },
                },
                select: { id: true },
            });
        }

        return bookToken;
    }

    async deleteBookTokenSubscriptionById(bookTokenId: string) {
        const isBookTokenSubscriptionExist = await this.prisma.bookTokenSubscription.findUnique({
            where: {
                id: bookTokenId,
            },
            select: {
                id: true,
                memberId: true,
                package: {
                    select: this.prisma.createSelect(['numberOfBookToken']),
                },
            },
        });

        if (!isBookTokenSubscriptionExist) {
            throw new HttpException('api-messages:book-token-not-found', HttpStatus.NOT_FOUND);
        }

        const updateRequest = await this.prisma.bookTokenSubscription.update({
            where: {
                id: bookTokenId,
            },
            data: {
                deletedAt: dayjs().toDate(),
            },
        });

        return updateRequest;
    }
}
