import { Body, Controller, Delete, Get, Param, Post, Put, Query, UploadedFile, UseInterceptors } from '@nestjs/common';
import { StaffBookTokenService } from './book-token.service';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { Authenticator, Authorizer, Staff } from 'src/auth/auth.decorator';
import { StaffInRequest, User } from 'src/types';
import { BookTokenIdParam, BookTokenParams, BookTokenQuery, UpdateBookTokenDto } from './book-token.dto';
import { bookTokenIdSchema, bookTokenParamsSchema, bookTokenQuerySchema, updateBookTokenSubscriptionByIdSchema } from './book-token.validator';
import { FileInterceptor } from '@nestjs/platform-express';
import { ActivityLogService } from 'src/activity-log/activity-log.service';

@Controller()
export class StaffBookTokenController {
    constructor(
        private readonly bookTokenService: StaffBookTokenService,
        private activityLogService: ActivityLogService,
    ) {}

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_TOKEN_VIEW')
    @Get()
    async getBookTokenSubscriptionList(@Query(new JoiValidationPipe(bookTokenQuerySchema)) query: BookTokenQuery) {
        const response = this.bookTokenService.getBookTokenSubscriptionList(query);

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_TOKEN_CREATE')
    @Post('upload')
    @UseInterceptors(FileInterceptor('file'))
    async uploadSubscriptionFile(@UploadedFile() file: Express.Multer.File) {
        const response = await this.bookTokenService.uploadPaymentProofMedia(file);

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_TOKEN_VIEW')
    @Get('/export')
    async getExportBookTokenSubscriptionList(@Query(new JoiValidationPipe(bookTokenParamsSchema)) query: BookTokenParams) {
        const response = this.bookTokenService.getExportBookTokenSubscriptionList(query);
        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_TOKEN_VIEW')
    @Get('/:bookTokenId')
    async getBookTokenSubscriptionById(@Param(new JoiValidationPipe(bookTokenIdSchema)) param: BookTokenIdParam) {
        const response = this.bookTokenService.getBookTokenSubscriptionById(param.bookTokenId);
        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_TOKEN_UPDATE')
    @Put('/:bookTokenId')
    async updateBookTokenSubscriptionById(
        @Staff() staff: StaffInRequest,
        @Param(new JoiValidationPipe(bookTokenIdSchema)) param: BookTokenIdParam,
        @Body(new JoiValidationPipe(updateBookTokenSubscriptionByIdSchema)) body: UpdateBookTokenDto,
    ) {
        const { id: staffId, fullName } = staff;
        const response = await this.bookTokenService.updateBookTokenSubscriptionById(param.bookTokenId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'book_token_subscription',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-book-token-subscription',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_TOKEN_DELETE')
    @Delete('/:bookTokenId')
    async deleteBookTokenSubscriptionById(@Param(new JoiValidationPipe(bookTokenIdSchema)) param: BookTokenIdParam) {
        const response = this.bookTokenService.deleteBookTokenSubscriptionById(param.bookTokenId);
        return response;
    }
}
