import { Controller, Get, Param } from '@nestjs/common';
import { BillService } from './bill.service';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { bookOrderInvoiceParamsSchema, invoiceParamsSchema } from './bill.validator';
import { BookOrderInvoiceParams, InvoiceParams } from './bill.dto';
import { Authenticator, Member } from 'src/auth/auth.decorator';
import { MemberInRequest, User } from 'src/types';

@Controller()
export class BillController {
    constructor(private billService: BillService) {}

    @Authenticator(User.STAFF)
    @Get('quotation/staff')
    async getStaffQuotationById() {
        const response = await this.billService.getStaffQuotationById();

        return response;
    }

    @Authenticator(User.MEMBER)
    @Get('quotation/member')
    async getMemberQuotationById(@Member() member: MemberInRequest) {
        const response = await this.billService.getMemberQuotationById(member.id);

        return response;
    }

    @Get('invoice/:bookTokenSubscriptionId')
    async getInvoiceByBookTokenSubscriptionId(@Param(new JoiValidationPipe(invoiceParamsSchema)) params: InvoiceParams) {
        const response = await this.billService.getInvoiceBySubscriptionId(params.bookTokenSubscriptionId);

        return response;
    }

    @Authenticator(User.STAFF)
    @Get('quotation/staff/book-order')
    async getBookQuotationById() {
        const response = await this.billService.getStaffBookQuotationById();

        return response;
    }

    @Get('invoice/:bookOrderId/book-order')
    async getInvoiceByBookOrderId(@Param(new JoiValidationPipe(bookOrderInvoiceParamsSchema)) params: BookOrderInvoiceParams) {
        const response = await this.billService.getInvoiceByBookOrderId(params.bookOrderId);

        return response;
    }

    @Authenticator(User.MEMBER)
    @Get('quotation/member/book-order')
    async getMemberBookQuotationById() {
        const response = await this.billService.getMemberBookQuotationById();

        return response;
    }
}
