import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import * as dayjs from 'dayjs';
import { EmailService } from 'src/email/email.service';
import { PrismaService } from 'src/prisma/prisma.service';
import { CustomException } from 'src/utils/CustomException';
import { v4 as uuidv4 } from 'uuid';

@Injectable()
export class MemberSharedService {
    constructor(
        private prisma: PrismaService,
        private emailService: EmailService,
    ) {}

    async emailCheck(
        email: string,
        excludedId?: string,
    ): Promise<{
        isCreated: boolean;
        isVerified: boolean;
    }> {
        const member = await this.prisma.member.findFirst({
            where: {
                email: {
                    contains: email,
                    mode: 'insensitive',
                },
                deletedAt: null,
                id: !!excludedId
                    ? {
                          not: excludedId,
                      }
                    : undefined,
            },
        });

        if (!member) {
            // Member does not exist
            return {
                isCreated: false,
                isVerified: false,
            };
        }

        if (!member.password) {
            // Member exists but not verified
            return {
                isCreated: true,
                isVerified: false,
            };
        }

        // Member exists and verified
        return {
            isCreated: true,
            isVerified: true,
        };
    }

    async createMember(body: CreateMemberDto) {
        // Check existing email
        const { isCreated } = await this.emailCheck(body.email);

        // If email already exists, throw error
        if (isCreated) {
            throw new CustomException('api-messages:email-already-exists', HttpStatus.CONFLICT, {
                isCreated: true,
            });
        }

        // Create member
        const member = await this.prisma.member.create({
            data: {
                email: body.email,
                fullName: body.fullName,
                preferredName: body.preferredName,
                phoneNumber: body.phoneNumber,
                lastActiveAt: dayjs().toDate(),
                address: body.address,
                dateOfBirth: body.dateOfBirth,
            },
            select: {
                ...this.prisma.createSelect(['id', 'email', 'fullName', 'preferredName', 'phoneNumber', 'address', 'dateOfBirth']),
            },
        });

        // Send verification email
        const emailResponse = await this.sendVerificationEmail(member.id, member.email, member.fullName);

        // If email failed to send, throw error
        if (!emailResponse.success) {
            throw new HttpException('api-messages:email-failed-to-send', HttpStatus.INTERNAL_SERVER_ERROR);
        }

        return member;
    }

    async sendVerificationEmail(memberId: string, email: string, fullName: string) {
        // Ensure the verification request is not more than 5 times per 12 hours
        const memberTokens = await this.prisma.memberToken.findMany({
            where: {
                memberId,
                type: 'CONFIRMATION',
                createdAt: {
                    gte: dayjs().subtract(12, 'hours').toDate(),
                },
            },
        });

        if (memberTokens.length >= 5) {
            throw new CustomException('api-messages:too-many-verification-requests', HttpStatus.TOO_MANY_REQUESTS, {
                tooManyRequests: true,
            });
        }

        // Create new token
        const newToken = uuidv4();
        const newTokenExpiredAt = dayjs().add(3, 'days');

        await this.prisma.memberToken.create({
            data: {
                memberId,
                token: newToken,
                type: 'CONFIRMATION',
                expiredAt: newTokenExpiredAt.toDate(),
            },
        });

        // Send verification email
        const emailResponse = await this.emailService.memberVerification(email, {
            name: fullName,
            memberId,
            token: newToken,
        });

        return emailResponse;
    }

    async resendVerificationEmail(email: string) {
        const { isCreated, isVerified } = await this.emailCheck(email);

        // Ensure member is created but not yet verified
        if (!isCreated) {
            throw new CustomException('api-messages:member-not-found', HttpStatus.NOT_FOUND, {
                isCreated,
            });
        }
        if (isVerified) {
            throw new CustomException('api-messages:member-already-verified', HttpStatus.CONFLICT, {
                isCreated,
                isVerified,
            });
        }

        // Resend verification email
        const member = await this.prisma.member.findFirst({
            where: {
                email,
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
            },
        });

        const emailResponse = await this.sendVerificationEmail(member.id, member.email, member.fullName);

        // If email failed to send, throw error
        if (!emailResponse.success) {
            throw new HttpException('api-messages:email-failed-to-send', HttpStatus.INTERNAL_SERVER_ERROR);
        }

        return member;
    }
}
