import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { PrismaService } from 'src/prisma/prisma.service';
import { OrderBookDto, BookOrderQuery, UpdateBookOrderDto } from './book-order.dto';
import { conditionalReturn } from 'src/utils';
import { BookOrderStatus, Prisma } from '@prisma/client';
import * as dayjs from 'dayjs';
import { MediaService } from 'src/media/media.service';

@Injectable()
export class BookOrderService {
    constructor(
        private prisma: PrismaService,
        private media: MediaService,
    ) {}

    async getBookOrderByPagination(memberId: string, query: BookOrderQuery) {
        const { page, pageSize, sortField, sortOrder, bookName, status } = query;

        /* Where Options */
        const whereOptions: Prisma.BookOrderWhereInput = {
            deletedAt: null,
            memberId,
            ...conditionalReturn(!!bookName, {
                book: { name: { mode: 'insensitive', contains: bookName } },
            }),
            ...conditionalReturn(!!status, { status }),
        };

        const bookCount = await this.prisma.bookOrder.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(bookCount, page, pageSize);

        const bookList = await this.prisma.bookOrder.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: {
                ...this.prisma.createSelect(['id', 'displayId', 'status', 'quantity', 'bookPrice']),
                book: { select: this.prisma.createSelect(['id', 'name']) },
                member: { select: this.prisma.createSelect(['id', 'fullName', 'preferredName']) },
            },
        });

        return {
            count: bookCount,
            rows: bookList,
            page: currentPage,
        };
    }

    async getBookOrderById(memberId: string, bookOrderId: string) {
        const bookOrder = await this.prisma.bookOrder.findFirst({
            where: {
                id: bookOrderId,
                memberId,
                deletedAt: null,
            },
            select: {
                id: true,
                bookId: true,
                book: {
                    select: {
                        name: true,
                    },
                },
                memberId: true,
                member: {
                    select: {
                        fullName: true,
                    },
                },
                bookPrice: true,
                status: true,
                quantity: true,
                media: {
                    select: {
                        mediaId: true,
                        media: {
                            select: {
                                id: true,
                                name: true,
                                type: true,
                                key: true,
                            },
                        },
                    },
                },
            },
        });

        if (!bookOrder) {
            throw new HttpException('api-messages:book-order-not-found', HttpStatus.NOT_FOUND);
        }

        return bookOrder;
    }

    async orderBook(memberId: string, body: OrderBookDto) {
        const { bookId, quantity } = body;

        const book = await this.prisma.book.findFirst({
            where: {
                id: bookId,
                deletedAt: null,
            },
        });

        if (!book) {
            throw new HttpException('api-messages:book-not-found', HttpStatus.NOT_FOUND);
        }

        const bookOrder = await this.prisma.bookOrder.create({
            data: {
                bookId,
                memberId,
                quantity,
                bookPrice: book.price,
            },
        });

        const updatedBookOrder = await this.updateDisplayId(bookOrder.id);

        return updatedBookOrder;
    }

    async updateBookOrder(memberId: string, bookOrderId: string, body: UpdateBookOrderDto) {
        const { quantity, media } = body;

        const bookOrder = await this.prisma.bookOrder.findFirst({
            where: {
                id: bookOrderId,
                memberId,
                deletedAt: null,
            },
        });

        if (!bookOrder) {
            throw new HttpException('api-messages:book-order-not-found', HttpStatus.NOT_FOUND);
        }

        const updatedBookOrder = await this.prisma.bookOrder.update({
            where: {
                id: bookOrder.id,
            },
            data: {
                quantity,
                media: {
                    deleteMany: {},
                    createMany: {
                        data: media.map((media) => ({
                            mediaId: media.mediaId,
                        })),
                    },
                },
                status: BookOrderStatus.PENDING,
            },
            select: this.prisma.createSelect(['id']),
        });

        return updatedBookOrder;
    }

    async uploadBookOrderMedia(file: Express.Multer.File) {
        const fileResponse = await this.media.upload(file, { isPublic: false });

        const mediaResponse = await this.prisma.media.create({
            data: fileResponse,
            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
        });

        return mediaResponse;
    }

    async deleteBookOrder(memberId: string, bookOrderId: string) {
        const bookOrder = await this.prisma.bookOrder.findFirst({
            where: {
                id: bookOrderId,
                memberId,
                deletedAt: null,
            },
        });

        if (!bookOrder) {
            throw new HttpException('api-messages:book-order-not-found', HttpStatus.NOT_FOUND);
        }

        const deletedBookOrder = await this.prisma.bookOrder.update({
            where: {
                id: bookOrder.id,
            },
            data: {
                deletedAt: dayjs().toDate(),
            },
            select: this.prisma.createSelect(['id']),
        });

        return deletedBookOrder;
    }

    private async updateDisplayId(bookOrderId: string) {
        const response = await this.prisma.bookOrder.findFirst({
            where: {
                id: bookOrderId,
            },
            select: this.prisma.createSelect(['refNo']),
        });

        const ref = response.refNo.toString().padStart(2, '0');
        const displayId = `BO-${dayjs().format('YYMMDD')}-${ref}`;

        const updateResponse = await this.prisma.bookOrder.update({
            where: {
                id: bookOrderId,
            },
            data: {
                displayId,
            },
            select: this.prisma.createSelect(['id']),
        });

        return updateResponse;
    }
}
