import { MailerService } from '@nestjs-modules/mailer';
import { Injectable } from '@nestjs/common';
import { MEMBER_URL, STAFF_URL } from 'src/config';
import {
    BirthdayContext,
    EMAIL_TEMPLATES,
    MemberResetPasswordContext,
    MemberVerificationContext,
    ReminderContext,
    SendEmailConfig,
    StaffResetPasswordContext,
    StaffVerificationContext,
    TaskOngoingContext,
} from 'src/types';

@Injectable()
export class EmailService {
    constructor(private mailerService: MailerService) {}

    async staffVerification(to: string | string[], context: StaffVerificationContext) {
        const url = `${STAFF_URL}/verification/${context.staffId}/${context.token}`;
        const emailResponse = this.sendEmail({
            to,
            subject: 'Staff Verification',
            template: EMAIL_TEMPLATES.STAFF_VERIFICATION,
            context: {
                ...context,
                url,
            },
        });
        return emailResponse;
    }

    async memberVerification(to: string | string[], context: MemberVerificationContext) {
        const url = `${MEMBER_URL}/verification/${context.memberId}/${context.token}`;
        const emailResponse = this.sendEmail({
            to,
            subject: 'Member Verification',
            template: EMAIL_TEMPLATES.MEMBER_VERIFICATION,
            context: {
                ...context,
                url,
            },
        });
        return emailResponse;
    }

    async staffResetPassword(to: string | string[], context: StaffResetPasswordContext) {
        const url = `${STAFF_URL}/reset-password/${context.staffId}/${context.resetToken}`;
        return this.sendEmail({
            to,
            subject: 'Reset Password',
            template: EMAIL_TEMPLATES.STAFF_RESET_PASSWORD,
            context: {
                ...context,
                url,
            },
        });
    }

    async memberResetPassword(to: string | string[], context: MemberResetPasswordContext) {
        const url = `${MEMBER_URL}/reset-password/${context.memberId}/${context.resetToken}`;
        return this.sendEmail({
            to,
            subject: 'Reset Password',
            template: EMAIL_TEMPLATES.MEMBER_RESET_PASSWORD,
            context: {
                ...context,
                url,
            },
        });
    }

    async taskOngoing(to: string | string[], context: TaskOngoingContext) {
        return this.sendEmail({
            to,
            subject: 'Task Ongoing',
            template: EMAIL_TEMPLATES.TASK_ONGOING,
            context,
        });
    }

    async birthday(to: string | string[], context: BirthdayContext) {
        return this.sendEmail({
            to,
            subject: 'Happy Birthday',
            template: EMAIL_TEMPLATES.BIRTHDAY,
            context,
        });
    }

    async reminder(to: string | string[], context: ReminderContext) {
        return this.sendEmail({
            to,
            subject: 'Reminder',
            template: EMAIL_TEMPLATES.REMINDER,
            context,
        });
    }

    private async sendEmail(config: SendEmailConfig) {
        const { to, subject, template, context, attachments } = config;

        try {
            const emailResponse = await this.mailerService.sendMail({
                to,
                subject,
                template,
                context,
                attachments,
            });

            if (emailResponse.rejected.length > 0) {
                // Do some logging here
            }
        } catch (error) {
            console.log(error);
            return Promise.resolve({
                success: false,
                message: error,
            });
        }

        return {
            success: true,
        };
    }
}
