import { Injectable } from '@nestjs/common';
import { PrismaService } from './prisma/prisma.service';
import * as dayjs from 'dayjs';
import { Cron, CronExpression } from '@nestjs/schedule';
import { GroupMember, GroupedMember, Task } from './types';
import { toNumber } from 'lodash';
import { MedicalCertificateStatus, StudyGroupStatus } from '@prisma/client';
import { EmailService } from './email/email.service';
import { MEMBER_URL } from './config';

@Injectable()
export class CronService {
    constructor(
        private prisma: PrismaService,
        private emailService: EmailService,
    ) {}

    @Cron(CronExpression.EVERY_DAY_AT_8PM)
    async sendReminder() {
        const studyGroupTask = await this.prisma.studyGroupTask.findMany({
            where: {
                deletedAt: null,
                taskDate: dayjs().startOf('day').toDate(),
                studyGroup: {
                    status: StudyGroupStatus.ONGOING,
                    deletedAt: null,
                }
            },
            select: {
                id: true,
                studyGroup: {
                    select: {
                        name: true,
                        id: true,
                        studyGroupMembers: {
                            select: {
                                member: {
                                    select: {
                                        id: true,
                                        fullName: true,
                                        email: true,
                                    },
                                },
                            },
                        },
                    },
                }
            },
        });

        // Check if the member has commented on the task
        for (const task of studyGroupTask) {
            for (const member of task.studyGroup.studyGroupMembers) {
                const comment = await this.prisma.studyGroupTaskComment.findFirst({
                    where: {
                        deletedAt: null,
                        studyGroupTaskId: task.id,
                        memberId: member.member.id,
                    },
                });

                if (!comment) {
                    console.log(`Reminder Sent to ${member.member.fullName}`.green.bold);
                    await this.emailService.reminder(member.member.email, {
                        name: member.member.fullName,
                        studyGroup: task.studyGroup.name,
                        studyGroupLink: `${MEMBER_URL}/study-group/${task.studyGroup.id}`,
                    });
                }
            }
        }
    }

    @Cron('0 01 0 * * *')
    async calculatePenalties() {
        const startDate = dayjs().subtract(1, 'day').startOf('day').toDate();
        const endDate = dayjs().subtract(1, 'day').endOf('day').toDate();

        /* Get all tasks and associated member comments */
        const tasks = await this.prisma.studyGroupTask.findMany({
            where: {
                deletedAt: null,
                taskDate: {
                    gte: startDate,
                    lte: endDate,
                },
            },
            select: {
                id: true,
                studyGroupId: true,
                studyGroupTaskComment: {
                    where: {
                        parent_id: null,
                        deletedAt: null,
                    },
                    select: {
                        memberId: true,
                    },
                },
            },
        });

        // Transform tasks data for easier processing
        const tasksData = this.taskTransformer(tasks);

        const uncommentedMember = [];

        for (const task of tasks) {
            // Get all members associated with the task's study group
            const members = await this.prisma.studyGroupMember.findMany({
                where: {
                    deletedAt: null,
                    studyGroupId: task.studyGroupId,
                    member: {
                        deletedAt: null,
                    },
                },
                select: {
                    studyGroupId: true,
                    memberId: true,
                },
            });

            // Transform members data for easier processing
            const transformedData = this.groupMemberTransformer(members);

            for (const member of transformedData) {
                const associatedTask = tasksData.find((t) => t.studyGroupId === member.studyGroupId);
                if (!associatedTask) continue;

                const memberIds = member.memberIds.filter((memberId) => {
                    const isCommented = associatedTask.memberIds.find((comment) => comment.memberId === memberId);
                    return !isCommented;
                });

                // Check medical certificate status for each member
                for (const memberId of memberIds) {
                    const medicalCertificate = await this.prisma.medicalCertificate.findFirst({
                        where: {
                            memberId,
                            status: MedicalCertificateStatus.APPROVED,
                            applyDate: startDate,
                            deletedAt: null,
                        },
                    });

                    if (!medicalCertificate) {
                        uncommentedMember.push({
                            memberId,
                            taskId: associatedTask.taskId,
                        });
                    }
                }
            }
        }

        if (uncommentedMember.length > 0) {
            const penaltyAmount = await this.getPenaltyAmount();
            for (const data of uncommentedMember) {
                const isPenaltyExist = await this.prisma.penalty.findFirst({
                    where: {
                        deletedAt: null,
                        studyGroupTaskId: data.taskId,
                        memberId: data.memberId,
                    },
                });

                if (!isPenaltyExist) {
                    await this.prisma.penalty.create({
                        data: {
                            memberId: data.memberId,
                            studyGroupTaskId: data.taskId,
                            amount: penaltyAmount,
                        },
                    });
                }
            }
        }

        console.log(`Penalties Calculated for ${dayjs().subtract(1, 'day').format('DD/MM/YYYY')}`.blue.bold);
        return uncommentedMember;
    }

    @Cron(CronExpression.EVERY_DAY_AT_MIDNIGHT)
    async checkStudyGroupStartDate() {
        const startDate = dayjs().startOf('day').toDate();

        const studyGroups = await this.prisma.studyGroup.findMany({
            where: {
                deletedAt: null,
                startDate: {
                    lte: startDate,
                },
                status: {
                    in: [StudyGroupStatus.GENERATED],
                },
            },
            select: {
                id: true,
                name: true,
                studyGroupMembers: {
                    select: {
                        memberId: true,
                        member: {
                            select: {
                                id: true,
                                fullName: true,
                                email: true,
                            },
                        },
                    },
                },
            },
        });

        if (studyGroups.length > 0) {
            await this.prisma.studyGroup.updateMany({
                where: {
                    id: {
                        in: studyGroups.map((studyGroup) => studyGroup.id),
                    },
                },
                data: {
                    status: 'ONGOING',
                },
            });

            // Send email to all members
            for (const studyGroup of studyGroups) {
                const members = studyGroup.studyGroupMembers;

                for (const member of members) {
                    await this.emailService.taskOngoing(member.member.email, {
                        name: member.member.fullName,
                        task: studyGroup.name,
                    });
                }
            }
        }

        console.log(`Study Group Start Date Checked for ${dayjs().subtract(1, 'day').format('DD/MM/YYYY')}`.blue.bold);
    }

    @Cron(CronExpression.EVERY_DAY_AT_MIDNIGHT)
    async checkStudyGroupEndDate() {
        const endDate = dayjs().subtract(1, 'day').endOf('day').toDate();

        const studyGroups = await this.prisma.studyGroup.findMany({
            where: {
                deletedAt: null,
                endDate: {
                    lte: endDate,
                },
                status: {
                    in: [StudyGroupStatus.ONGOING],
                },
            },
            select: {
                id: true,
            },
        });

        if (studyGroups.length > 0) {
            await this.prisma.studyGroup.updateMany({
                where: {
                    id: {
                        in: studyGroups.map((studyGroup) => studyGroup.id),
                    },
                },
                data: {
                    status: 'COMPLETED',
                },
            });
        }

        console.log(`Study Group End Date Checked for ${dayjs().subtract(1, 'day').format('DD/MM/YYYY')}`.blue.bold);
    }

    @Cron(CronExpression.EVERY_DAY_AT_MIDNIGHT)
    async checkBookTokenSubscriptionExpiryDate() {
        const endDate = dayjs().subtract(1, 'day').endOf('day').toDate();

        const bookTokenSubscriptions = await this.prisma.bookTokenSubscription.findMany({
            where: {
                deletedAt: null,
                expiredAt: {
                    lte: endDate,
                },
            },
            select: {
                memberId: true,
            },
        });

        // set member's book token to 0 when subscription expired
        if (bookTokenSubscriptions.length > 0) {
            await this.prisma.member.updateMany({
                where: {
                    id: {
                        in: bookTokenSubscriptions.map((subscription) => subscription.memberId),
                    },
                },
                data: {
                    bookTokens: 0,
                },
            });
        }

        console.log(`Book Token Subscription Expiry Date Checked for ${dayjs().subtract(1, 'day').format('DD/MM/YYYY')}`.blue.bold);
    }

    @Cron(CronExpression.EVERY_DAY_AT_8AM)
    async postPrePlannedComment() {
        await this.prisma.studyGroupTaskComment.updateMany({
            where: {
                deletedAt: null,
                prePlanned: true,
                prePlannedDate: {
                    lte: dayjs().toDate(),
                },
            },
            data: {
                prePlanned: false,
                commentDate: dayjs().toDate(),
                prePlannedDate: dayjs().toDate(),
            },
        });

        console.log(`Pre-Planned Comment Posted for ${dayjs().format('DD/MM/YYYY')}`.blue.bold);
    }

    @Cron(CronExpression.EVERY_DAY_AT_MIDNIGHT)
    async checkMemberBirthday() {
        const members = await this.prisma.member.findMany({
            where: {
                deletedAt: null,
            },
            select: {
                id: true,
                dateOfBirth: true,
                email: true,
                fullName: true,
            },
        });

        const checkMembers = members.filter((member) => {
            return dayjs(member.dateOfBirth).format('DD/MM') === dayjs().format('DD/MM');
        });

        if (members.length > 0) {
            for (const member of checkMembers) {
                await this.emailService.birthday(member.email, {
                    name: member.fullName,
                });
            }
        }

        console.log(`Member Birthday Checked for ${dayjs().format('DD/MM/YYYY')}`.blue.bold);
    }

    private taskTransformer(data: Task[]) {
        return data.map((task) => {
            return {
                studyGroupId: task.studyGroupId,
                taskId: task.id,
                memberIds: task.studyGroupTaskComment.map((comment) => {
                    return {
                        memberId: comment.memberId,
                    };
                }),
            };
        });
    }

    private groupMemberTransformer(data: GroupMember[]) {
        const groupedData: GroupedMember[] = [];

        data.forEach((member) => {
            const { studyGroupId, memberId } = member;

            let group = groupedData.find((item) => item.studyGroupId === studyGroupId);

            if (!group) {
                group = {
                    studyGroupId: studyGroupId,
                    memberIds: [],
                };
                groupedData.push(group);
            }

            group.memberIds.push(memberId);
        });

        return groupedData;
    }

    private async getPenaltyAmount() {
        const amount = await this.prisma.metaData.findFirst({
            where: {
                key: 'penalty',
            },
            select: {
                value: true,
            },
        });

        return toNumber(amount?.value ?? 0);
    }
}
