import { CompanyInfoModule } from './public/company-info/company-info.module';
import { MemberBookTokenModule } from './member/book-token/book-token.module';
import { StaffDashboardModule } from './staff/dashboard/dashboard.module';
import { CronService } from './cron.service';
import { BillModule } from './public/bill/bill.module';
import { MemberStudyGroupModule } from './member/study-group/study-group.module';
import { StaffStudyGroupModule } from './staff/study-group/study-group.module';
import { SiteSettingModule } from './staff/site-setting/site-setting.module';
import { MemberModule } from './member/member/member.module';
import { StaffStudyRequestModule } from './staff/study-request/study-request.module';
import { StaffBookModule } from './staff/books/book.module';
import { MemberBookModule } from './member/book/book.module';
import { StaffModule } from './staff/staff/staff.module';
import { Module } from '@nestjs/common';
import { ConfigModule, ConfigService } from '@nestjs/config';
import { JwtModule } from '@nestjs/jwt';
import { PrismaModule } from './prisma/prisma.module';
import { APP_FILTER, APP_GUARD, RouterModule } from '@nestjs/core';
import { CustomExceptionFilter, GlobalExceptionFilter } from './utils/CustomException';
import { ThrottlerGuard, ThrottlerModule, seconds } from '@nestjs/throttler';
import { EmailModule } from './email/email.module';
import { MediaModule } from './media/media.module';
import { StaffAuthModule } from './staff/auth/auth.module';
import { AuthenticationGuard, PermissionsGuard } from './auth/auth.guard';
import { StaffSharedModule } from './staff/staff-shared.module';
import { StaffDataModule } from './staff/data/data.module';
import { DataModule } from './data/data.module';
import { CacheModule } from './cache/cache.module';
import { CacheModule as CacheManagerModule } from '@nestjs/cache-manager';
import { StaffMemberModule } from './staff/member/member.module';
import { MemberSharedModule } from './member/member-shared.module';
import { StaffRoleModule } from './staff/role/role.module';
import { MemberAuthModule } from './member/auth/auth.module';
import { MemberMyProfileModule } from './member/my-profile/my-profile.module';
import { StaffMyProfileModule } from './staff/my-profile/my-profile.module';
import { ScheduleModule } from '@nestjs/schedule';
import { StaffBookTokenModule } from './staff/book-token/book-token.module';
import { MemberDataModule } from './member/data/data.module';
import { MedicalCertificateModule } from './staff/medical-certificate/medical-certificate.module';
import { ActivityLogModule } from './activity-log/activity-log.module';
import { StaffBookOrderModule } from './staff/book-order/book-order.module';
import { MemberBookOrderModule } from './member/book-order/book-order.module';

@Module({
    imports: [
        PrismaModule,
        EmailModule,
        MediaModule,
        DataModule,
        CacheManagerModule.register({
            ttl: 1000 * 60 * 5, // 5 minutes in milliseconds
            isGlobal: true,
        }),
        CacheModule,

        //Staff
        StaffDashboardModule,
        StaffModule,
        StaffMemberModule,
        StaffAuthModule,
        StaffSharedModule,
        StaffDataModule,
        StaffRoleModule,
        StaffMyProfileModule,
        StaffStudyRequestModule,
        StaffBookModule,
        StaffBookOrderModule,
        SiteSettingModule,
        StaffStudyGroupModule,
        StaffBookTokenModule,
        MedicalCertificateModule,

        //Member
        MemberModule,
        MemberAuthModule,
        MemberSharedModule,
        MemberBookModule,
        MemberBookOrderModule,
        MemberMyProfileModule,
        MemberStudyGroupModule,
        MemberBookTokenModule,
        MemberDataModule,

        // Public
        BillModule,
        ActivityLogModule,
        CompanyInfoModule,

        ScheduleModule.forRoot(),
        ConfigModule.forRoot({ isGlobal: true, envFilePath: `.env.${process.env.NODE_ENV ?? 'development'}` }),
        JwtModule.registerAsync({
            global: true,
            imports: [ConfigModule],
            useFactory: async (configService: ConfigService) => ({
                secret: configService.get('TOKEN_SECRET'),
                signOptions: {
                    expiresIn: '3days',
                },
            }),
            inject: [ConfigService],
        }),
        ThrottlerModule.forRoot([
            {
                ttl: seconds(60),
                limit: 60,
            },
        ]),
        RouterModule.register([
            {
                path: 'media',
                module: MediaModule,
            },
            {
                path: 'public',
                children: [
                    {
                        path: 'bill',
                        module: BillModule,
                    },
                    {
                        path: 'company-info',
                        module: CompanyInfoModule,
                    },
                ],
            },
            {
                path: 'activity-log',
                module: ActivityLogModule,
            },
            {
                path: 'staff',
                children: [
                    {
                        path: 'auth',
                        module: StaffAuthModule,
                    },
                    {
                        path: 'staff',
                        module: StaffModule,
                    },
                    {
                        path: 'member',
                        module: StaffMemberModule,
                    },
                    {
                        path: 'data',
                        module: StaffDataModule,
                    },
                    {
                        path: 'role',
                        module: StaffRoleModule,
                    },
                    {
                        path: 'book',
                        module: StaffBookModule,
                    },
                    {
                        path: 'book-order',
                        module: StaffBookOrderModule,
                    },
                    {
                        path: 'study-request',
                        module: StaffStudyRequestModule,
                    },
                    {
                        path: 'study-group',
                        module: StaffStudyGroupModule,
                    },
                    {
                        path: 'medical-certificate',
                        module: MedicalCertificateModule,
                    },
                    {
                        path: 'my-profile',
                        module: StaffMyProfileModule,
                    },
                    {
                        path: 'site-setting',
                        module: SiteSettingModule,
                    },
                    {
                        path: 'dashboard',
                        module: StaffDashboardModule,
                    },
                    {
                        path: 'book-token',
                        module: StaffBookTokenModule,
                    },
                ],
            },
            {
                path: 'member',
                children: [
                    {
                        path: 'auth',
                        module: MemberAuthModule,
                    },
                    {
                        path: 'book',
                        module: MemberBookModule,
                    },
                    {
                        path: 'book-order',
                        module: MemberBookOrderModule,
                    },
                    {
                        path: 'my-profile',
                        module: MemberMyProfileModule,
                    },
                    {
                        path: 'member',
                        module: MemberModule,
                    },
                    {
                        path: 'study-group',
                        module: MemberStudyGroupModule,
                    },
                    {
                        path: 'book-token',
                        module: MemberBookTokenModule,
                    },
                    {
                        path: 'data',
                        module: MemberDataModule,
                    },
                ],
            },
        ]),
    ],
    controllers: [],
    providers: [
        CronService,
        {
            provide: APP_GUARD,
            useClass: AuthenticationGuard,
        },
        {
            provide: APP_GUARD,
            useClass: PermissionsGuard,
        },
        {
            provide: APP_GUARD,
            useClass: ThrottlerGuard,
        },
        {
            provide: APP_FILTER,
            useClass: GlobalExceptionFilter,
        },
        {
            provide: APP_FILTER,
            useClass: CustomExceptionFilter,
        },
    ],
})
export class AppModule {}
